% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/selectSpecies.R
\name{selectSpecies}
\alias{selectSpecies}
\title{Select species based on traits}
\usage{
selectSpecies(
  t2c = NULL,
  constraints = NULL,
  t2d,
  obj = c("QH", "Q", "H"),
  phi = 0.5,
  capd = FALSE,
  euclid = TRUE
)
}
\arguments{
\item{t2c}{"Traits to constrain": A matrix of species trait values, where species are organized as rows and traits as columns. These should be numeric arrays, and no missing values are tolerated. Ordinal data will yield meaningful results, but nominal data entered as numeric vectors are invalid. Row and column names are optional. This matrix can be any dimension, but the number of traits should be less than the number of species.}

\item{constraints}{Trait constraints: A vector of trait values that serve as constants in the constraint equations. If not specified, the function will not constrain solutions to trait means. These trait contraints must be listed in the same order as columns in t2c. These constraints are community-weighted mean trait values, i.e., average traits weighted by the relative abundance of each species.}

\item{t2d}{"Traits to diversify": Can be 1) a matrix of species trait values to diversify where species are organized as rows and traits as columns. NAs are tolerated as long as each pair of species can be compared by at least one trait. In this case, dissimilarities among species are computed using Euclidean distance. The number of species in t2d must match those in t2c. Or 2) a distance matrix of class 'dist' that contains dissimilarities among species, no NAs are tolerated in the distance matrix.}

\item{obj}{Objective function: The objective function to optimise, one of three possibilities = c("QH", "Q", "H"). QH = Quadratic entropy (Q) plus entropy (H'); Q = Quadratic entropy; H = entropy.}

\item{phi}{A parameter bounded between 0 and 1 that weights the importance of either quadratic entropy or entropy (default = 0.5). Phi of 1 corresponds to 100 percent Q, phi of 0.5 corresponds to 50 percent Q and 50 perfect H', phi of 0 corresponds to 100 percent H'.}

\item{capd}{A logical stating whether the distance matrix should be capped at the mean distance among species (default = FALSE). Mean distance is calculated as the average of all upper triangular entries in the distance matrix calculated from t2d.}

\item{euclid}{A logical stating whether the distance matrix should be transformed to an Euclidean matrix if necessary (default = TRUE).}
}
\value{
A list with the elements: \item{prob}{Probabilities, i.e. optimal solutions of species relative abundance} \item{cwm_t2d}{Community-weighted mean trait values of resulting community for traits that were diversified, computed as probabilities x t2d using matrix multiplication} \item{cwm_t2c}{Community-weighted mean trait values of resulting community for traits that were constrained, computed as probabilities x t2d using matrix multiplication} \item{H}{Final entropy of community} \item{Q}{Final Rao Q of community} \item{objval}{Values of the objective function being maximized. The last value is the maximum.} \item{lagrange}{Lagrange multipliers.} \item{hessian}{The Hessian at the optimal solution.}
}
\description{
This function returns a probability distribution for a species pool based on their traits and a desired trait profile (Laughlin 2014). It can simultaneously constrain specific trait value(s) and optimise functional diversity.
}
\examples{
### 1 trait constraint with maximum entropy
Spp <- 5 #S = number of species
trait <- as.matrix(data.frame(trait=c(1:Spp)))
rownames(trait) <- c(letters[1:nrow(trait)])
result1 <- selectSpecies(t2c=trait, constraints=c(trait=3.5), t2d=trait, obj="H", capd=FALSE)
### compare result1 with virtually identical maxent output from FD package
#FD::maxent(constr=c(3.5),states=trait)$prob
### 1 trait constraint with maximum functional diversity
result2 <- selectSpecies(t2c=trait, constraints=c(trait=3.5), t2d=trait, obj="Q", capd=FALSE)
### 1 trait constraint with maximum functional diversity and entropy
result3 <- selectSpecies(t2c=trait, constraints=c(trait=3.5), t2d=trait, obj="QH", capd=FALSE)

### Plot results
plotProbs(result1,trait, xlab="Trait")
plotProbs(result2,trait, xlab="Trait")
plotProbs(result3,trait, xlab="Trait")

### 1 trait and no trait constraint
result4 <- selectSpecies(t2d=trait, obj="QH", capd=FALSE)
plotProbs(result4,trait, xlab="Trait")

##### 2 traits: Constrain trait X at X=3, diversify trait Y
traitX <- matrix(c(rep(1,4),rep(2,4),rep(3,4),rep(4,4)))
traitY <- matrix(c(rep(c(1,2,3,4),4)))
rownames(traitX) <- c(letters[1:16]); colnames(traitX) <- c("traitX")
rownames(traitY) <- c(letters[1:16]); colnames(traitY) <- c("traitY")

result5 <- selectSpecies(t2c=traitX,constraints=c(traitX=3),t2d=traitY,obj="Q",capd=FALSE)
result6 <- selectSpecies(t2c=traitX,constraints=c(traitX=3),t2d=traitY,obj="QH",capd=TRUE)

trait.matrix <- cbind(traitX, traitY)
plotProbs(result5,trait.matrix)
plotProbs(result6,trait.matrix)

##### 3 traits: Constrain trait Z to value 2.5, diversify trait X and Y
traitZ <- as.matrix(data.frame(c(1,3,2,2,3,1,2,3,1,2,1,3,2,3,2,2)))
rownames(traitZ) <- c(letters[1:16]); colnames(traitZ) <- c("traitZ")
result7 <- selectSpecies(t2c=traitZ,constraints=c(traitZ=2.5),t2d=trait.matrix, capd=TRUE, obj="QH")
plotProbs(result7,trait.matrix)

}
\references{
Laughlin, D.C. 2014. Applying trait-based models to achieve functional targets for theory-driven ecological restoration. Ecology Letters, 17, 771-784.
}
