% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SDForest.R
\name{SDForest}
\alias{SDForest}
\title{Spectrally Deconfounded Random Forests}
\usage{
SDForest(
  formula = NULL,
  data = NULL,
  x = NULL,
  y = NULL,
  nTree = 100,
  cp = 0,
  min_sample = 5,
  mtry = NULL,
  mc.cores = 1,
  Q_type = "trim",
  trim_quantile = 0.5,
  q_hat = 0,
  Qf = NULL,
  A = NULL,
  gamma = 7,
  max_size = NULL,
  return_data = TRUE,
  leave_out_ind = NULL,
  envs = NULL,
  nTree_leave_out = NULL,
  nTree_env = NULL,
  max_candidates = 100,
  Q_scale = TRUE,
  verbose = TRUE,
  predictors = NULL
)
}
\arguments{
\item{formula}{Object of class \code{formula} or describing the model to fit 
of the form \code{y ~ x1 + x2 + ...} where \code{y} is a numeric response and 
\code{x1, x2, ...} are vectors of covariates. Interactions are not supported.}

\item{data}{Training data of class \code{data.frame} containing the variables in the model.}

\item{x}{Matrix of covariates, alternative to \code{formula} and \code{data}.}

\item{y}{Vector of responses, alternative to \code{formula} and \code{data}.}

\item{nTree}{Number of trees to grow.}

\item{cp}{Complexity parameter, minimum loss decrease to split a node. 
A split is only performed if the loss decrease is larger than \code{cp * initial_loss}, 
where \code{initial_loss} is the loss of the initial estimate using only a stump.}

\item{min_sample}{Minimum number of observations per leaf. 
A split is only performed if both resulting leaves have at least 
\code{min_sample} observations.}

\item{mtry}{Number of randomly selected covariates to consider for a split, 
if \code{NULL} half of the covariates are available for each split. 
\eqn{\text{mtry} = \lfloor \frac{p}{2} \rfloor}}

\item{mc.cores}{Number of cores to use for parallel computation `vignette("Runtime")`. 
The `future` package is used for parallel processing. 
To use custom processing plans mc.cores has to be <= 1, see [`future` package](https://future.futureverse.org/).}

\item{Q_type}{Type of deconfounding, one of 'trim', 'pca', 'no_deconfounding'. 
'trim' corresponds to the Trim transform \insertCite{Cevid2020SpectralModels}{SDModels} 
as implemented in the Doubly debiased lasso \insertCite{Guo2022DoublyConfounding}{SDModels}, 
'pca' to the PCA transformation\insertCite{Paul2008PreconditioningProblems}{SDModels}. 
See \code{\link{get_Q}}.}

\item{trim_quantile}{Quantile for Trim transform, 
only needed for trim, see \code{\link{get_Q}}.}

\item{q_hat}{Assumed confounding dimension, only needed for pca, 
see \code{\link{get_Q}}.}

\item{Qf}{Spectral transformation, if \code{NULL} 
it is internally estimated using \code{\link{get_Q}}.}

\item{A}{Numerical Anchor of class \code{matrix}. See \code{\link{get_W}}.}

\item{gamma}{Strength of distributional robustness, \eqn{\gamma \in [0, \infty]}. 
See \code{\link{get_W}}.}

\item{max_size}{Maximum number of observations used for a bootstrap sample.
If \code{NULL} n samples with replacement are drawn.}

\item{return_data}{If \code{TRUE}, the training data is returned in the output.
This is needed for \code{\link{prune.SDForest}}, \code{\link{regPath.SDForest}}, 
and for \code{\link{mergeForest}}.}

\item{leave_out_ind}{Indices of observations that should not be used for training.}

\item{envs}{Vector of environments of class \code{factor} 
which can be used for stratified tree fitting.}

\item{nTree_leave_out}{Number of trees that should be estimated while leaving
one of the environments out. Results in number of environments times number of trees.}

\item{nTree_env}{Number of trees that should be estimated for each environment.
Results in number of environments times number of trees.}

\item{max_candidates}{Maximum number of split points that are 
proposed at each node for each covariate.}

\item{Q_scale}{Should data be scaled to estimate the spectral transformation? 
Default is \code{TRUE} to not reduce the signal of high variance covariates, 
and we do not know of a scenario where this hurts.}

\item{verbose}{If \code{TRUE} progress updates are shown using the `progressr` package. 
To customize the progress bar, see [`progressr` package](https://progressr.futureverse.org/articles/progressr-intro.html)}

\item{predictors}{Subset of colnames(X) or numerical indices of the covariates 
for which an effect on y should be estimated. All the other covariates are only
used for deconfounding.}
}
\value{
Object of class \code{SDForest} containing:
\item{predictions}{Vector of predictions for each observation.}
\item{forest}{List of SDTree objects.}
\item{var_names}{Names of the covariates.}
\item{oob_loss}{Out-of-bag loss. MSE}
\item{oob_SDloss}{Out-of-bag loss using the spectral transformation.}
\item{var_importance}{Variable importance.
The variable importance is calculated as the sum of the decrease in the loss function 
resulting from all splits that use a covariate for each tree. 
The mean of the variable importance of all trees results in the variable importance for the forest.}
\item{oob_ind}{List of indices of trees that did not contain the observation in the training set.}
\item{oob_predictions}{Out-of-bag predictions.}
If \code{return_data} is \code{TRUE} the following are also returned:
\item{X}{Matrix of covariates.}
\item{Y}{Vector of responses.}
\item{Q}{Spectral transformation.}
If \code{envs} is provided the following are also returned:
\item{envs}{Vector of environments.}
\item{nTree_env}{Number of trees for each environment.}
\item{ooEnv_ind}{List of indices of trees that did not contain the observation or the same environment in the training set
for each observation.}
\item{ooEnv_loss}{Out-of-bag loss using only trees that did not contain the observation or the same environment.}
\item{ooEnv_SDloss}{Out-of-bag loss using the spectral transformation and only trees that did not contain the observation
or the same environment.}
\item{ooEnv_predictions}{Out-of-bag predictions using only trees that did not contain the observation or the same environment.}
\item{nTree_leave_out}{If environments are left out, the environment for each tree, that was left out.}
\item{nTree_env}{If environments are provided, the environment each tree is trained with.}
}
\description{
Estimate regression Random Forest using spectral deconfounding.
The spectrally deconfounded Random Forest (SDForest) combines SDTrees in the same way, 
as in the original Random Forest \insertCite{Breiman2001RandomForests}{SDModels}.
The idea is to combine multiple regression trees into an ensemble in order to 
decrease variance and get a smooth function. Ensembles work best if the different 
models are independent of each other. To decorrelate the regression trees as much 
as possible from each other, we have two mechanisms. The first one is bagging 
\insertCite{Breiman1996BaggingPredictors}{SDModels}, where we train each regression 
tree on an independent bootstrap sample of the observations, e.g., we draw a 
random sample of size \eqn{n} with replacement from the observations. 
The second mechanic to decrease the correlation is that only a random subset 
of the covariates is available for each split. Before each split, 
we sample \eqn{\text{mtry} \leq p} from all the covariates and choose the one 
that reduces the loss the most only from those.
\deqn{\widehat{f(X)} = \frac{1}{N_{tree}} \sum_{t = 1}^{N_{tree}} SDTree_t(X)}
}
\examples{
set.seed(1)
n <- 50
X <- matrix(rnorm(n * 5), nrow = n)
y <- sign(X[, 1]) * 3 + rnorm(n)
model <- SDForest(x = X, y = y, Q_type = 'no_deconfounding', nTree = 5, cp = 0.5)
predict(model, newdata = data.frame(X))

###### subset of predictors
# if we know, that only the first covariate has an effect on y,
# we can estimate only its effect and use the others just for deconfounding
model <- SDForest(x = X, y = y, cp = 0.5, nTree = 5, predictors = c(1))

\donttest{
set.seed(42)
# simulation of confounded data
sim_data <- simulate_data_nonlinear(q = 2, p = 150, n = 100, m = 2)
X <- sim_data$X
Y <- sim_data$Y
train_data <- data.frame(X, Y)
# causal parents of y
sim_data$j

# comparison to classical random forest
fit_ranger <- ranger::ranger(Y ~ ., train_data, importance = 'impurity')

# you can customize the progress bar see parameter verbose
progressr::handlers("cli")
fit <- SDForest(x = X, y = Y, nTree = 100, Q_type = 'pca', q_hat = 2)
fit <- SDForest(Y ~ ., nTree = 100, train_data)
fit

# we can plot the fit to see whether the number of trees is high enough
# if the performance stabilizes, we have enough trees otherwise one can fit
# more and add them
plot(fit)

# a few more might be helpfull
progressr::handlers(progressr::handler_txtprogressbar(char = cli::col_red(cli::symbol$heart)))
fit2 <- SDForest(Y ~ ., nTree = 50, train_data) 
fit <- mergeForest(fit, fit2)

# comparison of variable importance
imp_ranger <- fit_ranger$variable.importance
imp_sdf <- fit$var_importance
imp_col <- rep('black', length(imp_ranger))
imp_col[sim_data$j] <- 'red'

plot(imp_ranger, imp_sdf, col = imp_col, pch = 20,
     xlab = 'ranger', ylab = 'SDForest', 
     main = 'Variable Importance')

# check regularization path of variable importance
path <- regPath(fit)
# out of bag error for different regularization
plotOOB(path)
plot(path)

# detection of causal parent using stability selection
stablePath <- stabilitySelection(fit)
plot(stablePath)

# pruning of forest according to optimal out-of-bag performance
fit <- prune(fit, cp = path$cp_min)

# partial functional dependence of y on the most important covariate
most_imp <- which.max(fit$var_importance)
dep <- partDependence(fit, most_imp)
plot(dep, n_examples = 100)
}
}
\references{
\insertAllCited{}
}
\seealso{
\code{\link{get_Q}}, \code{\link{get_W}}, \code{\link{SDTree}}, 
\code{\link{simulate_data_nonlinear}}, \code{\link{regPath}}, 
\code{\link{stabilitySelection}}, \code{\link{prune}}, \code{\link{partDependence}}
}
\author{
Markus Ulmer
}
