#' Generic Quantile-Quantile(Q-Q) Plot Function
#'
#' Generates a Q-Q (quantile–quantile) plot for any custom or built-in
#' probability distribution. The function compares sample quantiles with
#' theoretical quantiles computed using a user-specified quantile function.
#'
#' @param sample A numeric vector of sample observations.
#' @param qfun A quantile function corresponding to the theoretical distribution
#'   (e.g., \code{qnorm}, \code{qexp}, or a custom quantile function).
#' @param params A named list of distribution parameters
#'   (e.g., \code{list(mean = 0, sd = 1)} or \code{list(alpha = 2, lambda = 1)}).
#' @param fit.line Logical; if \code{TRUE} (default), a red least-squares
#'   regression line is added to the plot. Also, displays the regression
#' line equation and R-square value on the plot.
#'
#' @details
#' The function is general and can be used with any continuous distribution
#' for which a quantile function is available. It overlays both a 45° reference
#' line and (optionally) a fitted linear regression line through the points,
#' enabling visual assessment of model fit. Also, displays the regression
#' line equation and R² value on the plot. 
#'
#' Requires user-defined function \code{'qfun'} for the CDF of the 
#' user-defined continuous distribution.
#'
#' Missing values in the sample are automatically removed with a warning.
#'
#' @return
#' This function returns no value; it produces a Q-Q plot.
#'
#' @examples
#' # Example 1: Exponential distribution
#' set.seed(123)
#' x <- rexp(100, rate = 2)
#' qq.plot(x, qexp, list(rate = 2))
#'
#' # Example 2: Customizing the fitted line
#' qq.plot(x, qexp, list(rate = 2),
#'               fit.line = TRUE)
#'
#' # Example 3: Without regression line
#' qq.plot(x, qexp, list(rate = 2), fit.line = FALSE)
#'
#' # Example 4: Display regression equation and R-square value
#' qq.plot(x, qexp, list(rate = 2), fit.line = TRUE)
#'
#' # Example 5: For a user defined distribution
#' # Exponentiated Exponential Power (EEP) Distribution
#' #Data
#' x <- waiting
#' qq.plot(x,
#'       params = list(alpha=0.3407, lambda=0.6068, theta=7.6150),
#'       qfun = qgen.exp.power, fit.line=TRUE)
#' 
#' @export
qq.plot <- function(sample, qfun, params, fit.line=FALSE) {
  # Validate inputs
  if (!is.numeric(sample) || length(sample) < 2)
    stop("'sample' must be a numeric vector with at least two observations.")
  if (!is.function(qfun))
    stop("'qfun' must be a valid quantile function.")
  if (!is.list(params) || is.null(names(params)))
    stop("'params' must be a named list of distribution parameters.,
 	      e.g. list(alpha=1, lambda=2, theta=1.5).", call. = FALSE)

  sample <- na.omit(sample)
  n <- length(sample)
  
  probs <- ppoints(n)
  theoretical_q <- do.call(qfun, c(list(p = probs), params))
  
  sample_q <- sort(sample)
  
  plot(theoretical_q, sample_q,
       main = "Q-Q Plot",
       xlab = "Theoretical Quantiles",
       ylab = "Sample Quantiles",
       pch = 19, col = "blue", las=1)
  
  abline(0, 1, col = "red", lty = 1, lwd=2)
  
  if (isTRUE(fit.line)) {
	fit <- lm(sample_q ~ theoretical_q)
    abline(fit, col = "darkgreen", lwd = 2, lty = 2) 
	coef_fit <- coef(fit)
      r2 <- summary(fit)$r.squared
      eq <- bquote(y == .(round(coef_fit[1], 3)) + 
                       .(round(coef_fit[2], 3)) * x ~~ 
                       (R^2 == .(round(r2, 4))))
      legend("topleft", legend = as.expression(eq), bty = "n", text.col = "darkgreen")
  }
}


