# LogisInvExp.R
#' Logistic Inverse Exponential Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Logistic Inverse Exponential distribution.
#'
#' The Logistic Inverse Exponential distribution is parameterized by the parameters
#' \eqn{\alpha > 0} and \eqn{\lambda > 0}.
#'
#' @details
#' The Logistic Inverse Exponential distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \lambda) =
#' \quad \frac{1}{1+[\exp \{\lambda / x\}-1]^\alpha} 
#' \, ; \quad x > 0.
#' }
#'
#' where \eqn{\alpha} and \eqn{\lambda} are the parameters.
#'
#' Available functions are:
#' \itemize{
#'   \item \code{dlogis.inv.exp()} — Density function
#'   \item \code{plogis.inv.exp()} — Distribution function
#'   \item \code{qlogis.inv.exp()} — Quantile function
#'   \item \code{rlogis.inv.exp()} — Random generation
#'   \item \code{hlogis.inv.exp()} — Hazard function
#' }
#'
#' @usage
#' dlogis.inv.exp(x, alpha, lambda, log = FALSE)
#' plogis.inv.exp(q, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
#' qlogis.inv.exp(p, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
#' rlogis.inv.exp(n, alpha, lambda)
#' hlogis.inv.exp(x, alpha, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlogis.inv.exp}: numeric vector of (log-)densities
#'   \item \code{plogis.inv.exp}: numeric vector of probabilities
#'   \item \code{qlogis.inv.exp}: numeric vector of quantiles
#'   \item \code{rlogis.inv.exp}: numeric vector of random variates
#'   \item \code{hlogis.inv.exp}: numeric vector of hazard values
#' }
#'
#' @references    
#' Chaudhary, A.K., & Kumar, V. (2020). 
#' Logistic Inverse Exponential Distribution with Properties and Applications. 
#' \emph{International Journal of Mathematics Trends and Technology}, 
#' \bold{66(10)}, 151--162.  
#' \doi{10.14445/22315373/IJMTT-V66I10P518} 
#'  
#' @examples
#' x <- seq(0.1, 10, 0.5)
#' dlogis.inv.exp(x, 2.5, 1.5)
#' plogis.inv.exp(x, 2.5, 1.5)
#' qlogis.inv.exp(0.5, 2.5, 1.5)
#' rlogis.inv.exp(10, 2.5, 1.5)
#' hlogis.inv.exp(x, 2.5, 1.5)
#' 
#' # Data
#' x <- stress31
#' # ML estimates
#' params = list(alpha=7.6230, lambda=91.7136)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plogis.inv.exp, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlogis.inv.exp, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlogis.inv.exp, pfun=plogis.inv.exp, plot=FALSE)
#' print.gofic(out)
#'
#' @name LogisInvExp
#' @aliases dlogis.inv.exp plogis.inv.exp qlogis.inv.exp rlogis.inv.exp hlogis.inv.exp
#' @export
NULL

#' @rdname LogisInvExp
#' @usage NULL
#' @export   
dlogis.inv.exp <- function (x, alpha, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda) <= 0)) {
		stop("Parameters 'alpha' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
    u <- exp(lambda/x[valid])
    num <- lambda * alpha * u * ((u - 1.0)^(alpha - 1.0)) * (x[valid]^-2)
    deno <- (1.0 + (u - 1.0) ^ alpha) ^ 2.0
    pdf[valid] <- num/deno
	}
    if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname LogisInvExp
#' @usage NULL
#' @export 
plogis.inv.exp <- function (q, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda) <= 0)) {
		stop("Parameters 'alpha' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
    u <- exp(lambda/q[valid])
    tmp <- 1 + ((u - 1) ^ alpha)
    cdf[valid] <- 1/tmp
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)   
}

#' @rdname LogisInvExp
#' @usage NULL
#' @export 
qlogis.inv.exp <- function (p, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda) <= 0))
		stop("Invalid parameters: alpha, and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
    qtl[valid] <- lambda/(log(1 +(((1-p[valid])/p[valid]) ^ (1/alpha))))   
    }

    return(qtl)  
}

#' @rdname LogisInvExp
#' @usage NULL
#' @export 
rlogis.inv.exp <- function(n, alpha, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, lambda) <= 0))
		stop("Invalid parameters: alpha, and lambda must be positive.")

    u <- runif(n)
    q <- qlogis.inv.exp(u, alpha, lambda)
    return(q)
}

#' @rdname LogisInvExp
#' @usage NULL
#' @export 
hlogis.inv.exp <- function (x, alpha, lambda)
{
    if (!all(sapply(list(x, alpha, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
      
    nume <- dlogis.inv.exp(x, alpha, lambda) 
    surv <- 1.0 - plogis.inv.exp(x, alpha, lambda)
    hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)  
} 

