\name{E_ReMI}
\alias{E_ReMI}
\title{Two-Mode Clustering via the E_ReMI Method}
\description{
Performs row-by-column clustering using the E-ReMI (Extended REMAXINT) algorithm. 
It estimates latent cluster structures in rows and columns of a data matrix with unequal 
row cluster proportions using a REMAXINT-like procedure.
}
\usage{
E_ReMI(DC, P, Q, Nruns, initR, initC, initG)
}
\arguments{
  \item{DC}{A doubly centered numeric matrix of size \code{I x J} representing the input data to be clustered.}
  \item{P}{An integer indicating the number of clusters for rows.}
  \item{Q}{An integer indicating the number of clusters for columns.}
  \item{Nruns}{The number of random initializations (runs) to perform to avoid local maxima. 
               The solution with the highest likelihood is returned.}
  \item{initR}{An initial binary matrix of size \code{I x P} indicating the initial row cluster assignments.}
  \item{initC}{An initial binary matrix of size \code{J x Q} indicating the initial column cluster assignments.}
  \item{initG}{A \code{P x Q} matrix representing the initial estimate of the latent interaction 
               between row and column clusters.}
}
\details{
The \code{E_ReMI} function implements an EM-style algorithm for biclustering, 
based on maximizing the likelihood of a Gaussian interaction model with unequal row cluster sizes. 
Starting from user-provided initial values for cluster assignments and the interaction matrix, 
the algorithm iteratively updates row clusters, column clusters, the bi-cluster interaction matrix, 
and estimates the proportions of row clusters (\code{Omega}) and residual variance. 
Unlike REMAXINT clustering models, \code{E_ReMI} accounts for unequal cluster sizes in expectation, 
allowing for more flexible modeling of heterogeneous block-structured data.
}
\value{
A list containing:
  \item{BestR}{Binary matrix (\code{I x P}) of row cluster assignments.}
  \item{BestC}{Binary matrix (\code{J x Q}) of column cluster assignments.}
  \item{BestG}{Estimated \code{P x Q} latent interaction matrix.}
  \item{LL}{Maximal log-likelihood of the fitted model.}
  \item{Best_O}{Estimated row cluster proportions (Omega), a vector of length \code{P}.}
}
\references{
Ahmed, Z., van Breukelen, G. J. P., Schepers, J., & Cassese, A. (2025). Robustness study of normality-based likelihood ratio tests 
for testing maximal interaction two-mode clustering and a permutation-based alternative. \emph{Available on OSF} (submitted to ADAC).
Ahmed, Z., van Breukelen, G. J. P., Schepers, J., & Cassese, A. (2023). E-ReMI: extended maximal interaction two-mode clustering. 
\emph{Journal of Classification, 40, 298-331.}
Ahmed, Z., van Breukelen, G. J. P., Schepers, J., & Cassese, A. (2021). REMAXINT: a two-mode clustering-based method for statistical
inference on two-way interaction. \emph{Advances in Data Analysis and Classification, 15(4), 987-1013.}
}
\author{
Zaheer Ahmed (ahmed@ifado.de), 
Jan Schepers (jan.schepers@maastrichtuniversity.nl), 
Uwe Ligges (ligges@statistik.tu-dortmund.de), 
Alberto Cassese (alberto.cassese@unifi.it), 
Gerard van Breukelen (gerard.vbreukelen@maastrichtuniversity.nl), 
Katja Ickstadt (ickstadt@statistik.tu-dortmund.de), 
Edmund Wascher (wascher@ifado.de) 
}
\note{
This function is useful for clustering data with unequal row cluster sizes and estimating interaction structures between biclusters.
}
\seealso{
\code{\link{Permutation_Function}}, 
\code{\link{Randompartition_function}}, 
\code{\link{Unequal_Randompartition_Function}}, 
\code{\link{Update_row_clusters_E_ReMI}}, 
\code{\link{Update_column_clusters}}, 
\code{\link{Update_G_Omega}}, 
\code{\link{Log_Likelihood_function_E_ReMI}}, 
\code{\link[pracma]{pinv}}
}
\examples{
I <- 10
P <- 4
J <- 5 
Q <- 2
Nruns <- 5
DC <- matrix(rnorm(I * J), I, J)
initR <- diag(1, nrow = I, ncol = P)
initC <- diag(1, nrow = J, ncol = Q)
R_inv <- pracma::pinv(t(initR) \%*\% initR)
C_inv <- pracma::pinv(t(initC) \%*\% initC)
initG <- R_inv \%*\% t(initR) \%*\% DC \%*\% initC \%*\% C_inv
initG <- as.matrix(initG)
result <- E_ReMI(DC, P, Q, Nruns, initR, initC, initG)
result
}
\keyword{E-ReMI}
\keyword{Two-mode clustering}
\keyword{Maximal interaction}
