#' Chiclet differential deuterium uptake plot
#' 
#' @description Chiclet plot of differential deuterium uptake values 
#' between two biological states in time.
#' 
#' @param diff_uptake_dat data produced by \code{\link{calculate_diff_uptake}} or
#' \code{\link{create_diff_uptake_dataset}} function
#' @param diff_p_uptake_dat differential uptake data 
#' alongside the P-value as calculated by 
#' \code{\link{create_p_diff_uptake_dataset}}
#' @param theoretical \code{logical}, determines if values are theoretical
#' @param fractional \code{logical}, determines if values are fractional
#' @param show_uncertainty \code{logical}, determines if the
#' uncertainty is shown
#' @param show_houde_interval \code{logical}, determines if houde interval is shown
#' @param show_tstud_confidence \code{logical}, determines if t-Student test validity 
#' is shown
#' @param confidence_level confidence level for the test, from range [0, 1]
#' Important if selected show_confidence_limit
#' @inheritParams plot_butterfly
#'
#' @details Function \code{\link{plot_differential_chiclet}} generates
#' chiclet differential plot based on provided data and parameters.
#' On X-axis there is a peptide ID. On Y-axis are time points
#' of measurement. Each tile for a peptide in time has a color value
#' representing the deuterium uptake difference between chosen states,
#' in a form based on provided criteria (e.q. fractional). Each tile has
#' a plus sign, which size represent the uncertainty of measurement for
#' chosen value.
#'
#' @return a [ggplot2::ggplot()] object.
#'
#' @seealso
#' \code{\link{create_diff_uptake_dataset}}
#' \code{\link{calculate_diff_uptake}}
#'
#' @examples
#' diff_uptake_dat <- create_diff_uptake_dataset(alpha_dat)
#' plot_differential_chiclet(diff_uptake_dat)
#' plot_differential_chiclet(diff_uptake_dat, show_houde_interval = TRUE)
#' 
#' diff_p_uptake_dat <- create_p_diff_uptake_dataset(alpha_dat)
#' plot_differential_chiclet(diff_p_uptake_dat = diff_p_uptake_dat, 
#'                           show_tstud_confidence = TRUE)
#' plot_differential_chiclet(diff_p_uptake_dat = diff_p_uptake_dat, 
#'                           show_tstud_confidence = TRUE, show_houde_interval = TRUE) 
#' 
#' @export plot_differential_chiclet

plot_differential_chiclet <- function(diff_uptake_dat = NULL, 
                                      diff_p_uptake_dat = NULL, 
                                      theoretical = FALSE,
                                      fractional = FALSE,
                                      show_houde_interval = FALSE,
                                      show_tstud_confidence = FALSE,
                                      confidence_level = 0.98,
                                      show_uncertainty = FALSE,
                                      interactive = getOption("hadex_use_interactive_plots")){
  
  if (show_tstud_confidence) {
    
    if(is.null(diff_p_uptake_dat)) { stop("Please, provide the neccessary data.") } else { diff_uptake_dat <- diff_p_uptake_dat }
    
  } else {
    
    if(is.null(diff_uptake_dat) & is.null(diff_p_uptake_dat)) { stop("Please, provide the neccessary data.") } 
    
  }
  
  ##
  
  if (theoretical) {
    
    if (fractional) {
      
      # theoretical & fractional
      value <- "diff_theo_frac_deut_uptake"
      err_value <- "err_diff_theo_frac_deut_uptake"
      title <- "Theoretical chiclet differential plot"
      fill <- "Fractional DU Diff"
      unit <- "[%]"
      
    } else {
      
      # theoretical & absolute
      value <- "diff_theo_deut_uptake"
      err_value <- "err_diff_theo_deut_uptake"
      title <- "Theoretical chiclet differential plot"
      fill <- "DU Diff"
      unit <- "[Da]"
      
    }
    
  } else {
    
    if (fractional) {
      
      # experimental & fractional
      value <- "diff_frac_deut_uptake"
      err_value <- "err_diff_frac_deut_uptake"
      title <- "Chiclet differential plot"
      fill <- "Fractional DU Diff"
      unit <- "[%]"
      
    } else {
      
      # experimental & absolute
      value <- "diff_deut_uptake"
      err_value <- "err_diff_deut_uptake"
      title <- "Chiclet differential plot"
      fill <- "DU Diff"
      unit <- "[Da]"
    }
    
  }
  
  diff_uptake_dat <- as.data.table(diff_uptake_dat)
  
  plot_dat <- data.frame(ID = diff_uptake_dat[["ID"]],
                         Exposure = as.factor(diff_uptake_dat[["Exposure"]]),
                         value = diff_uptake_dat[[value]],
                         err_value = diff_uptake_dat[[err_value]],
                         Sequence = diff_uptake_dat[["Sequence"]],
                         Start = diff_uptake_dat[["Start"]],
                         End = diff_uptake_dat[["End"]])
  
  chosen_tile_geom <- if (interactive) ggiraph::geom_tile_interactive(
    aes(tooltip = glue(
      "{Sequence}
       Position: {Start}-{End}
       ID: {ID}
       Value: {round(value, 2)} {unit}
       Exposure: {Exposure} min"
    ))
  ) else geom_tile()
  
  attr(plot_dat, "n_rep") <- attr(diff_uptake_dat, "n_rep")
  
  min_du <- min(plot_dat[["value"]])
  max_du <- max(plot_dat[["value"]])
  
  chiclet_differential_plot <- ggplot(plot_dat, aes(y = Exposure, x = ID, fill = value)) +
    chosen_tile_geom +
    geom_tile(data = subset(plot_dat, is.na(value)), fill = "gray95") +
    scale_fill_gradient2(low = "blue", mid = "white", high = "red", guide = guide_legend(keywidth = 3), limits = c(min_du, max_du)) + #       limits = c(min(facetgrid$value), max(facetgrid$value)))
    labs(title = title,
         y = "Exposure [min]",
         x = "Peptide ID",
         fill = fill) +
    theme(legend.position = "bottom",
          legend.box = "vertical",
          legend.key = element_rect(colour = 'black', size = 1))
  
  if(show_uncertainty){
    
    chiclet_differential_plot <- chiclet_differential_plot +
      geom_point(aes(size = err_value), shape = 3) +
      labs(size = "Err")
    
  }
  
  if(show_houde_interval){
    
    # t_value <- qt(c((1 - confidence_level)/2, 1-(1 - confidence_level)/2), df = 2)[2]
    # x_threshold <- t_value * mean(plot_dat[["err_value"]], na.rm = TRUE)/sqrt(length(plot_dat))
    
    x_threshold <- calculate_confidence_limit_values(plot_dat, 
                                                     confidence_level = confidence_level,
                                                     n_rep = attr(diff_uptake_dat, "n_rep"))[2]
    
    chiclet_differential_plot <- chiclet_differential_plot +
      geom_tile(data = subset(plot_dat, abs(value) < x_threshold), fill = "grey91")
    
  }
  
  if(show_tstud_confidence){
    
    alpha <- -log(1 - confidence_level)
    
    #### datatable extra
    diff_uptake_dat <- data.table(diff_uptake_dat)
    
    diff_uptake_dat[, `:=`(valid = log_p_value >= alpha, 
                           Exposure = as.factor(Exposure))]
      
    diff_uptake_dat <- merge(diff_uptake_dat, plot_dat, by = c("Sequence", "Start", "End", "Exposure", "ID"))
    
    chiclet_differential_plot <- chiclet_differential_plot +
      geom_tile(data = subset(diff_uptake_dat, !valid), aes(x = ID, y = Exposure), fill = "grey89") +
      geom_tile(data = subset(diff_uptake_dat, is.na(valid)), aes(x = ID, y = Exposure), fill = "grey56")
    
  }
  
  return(HaDeXify(chiclet_differential_plot))
  
}
