\name{localdecodeHSMM}
\alias{localdecodeHSMM}
\title{Local Decoding for Hidden Semi-Markov Models}
\description{
Computes the most likely hidden state sequence for a univariate time series
using a fitted Hidden Semi-Markov Model (HSMM).
}
\usage{
localdecodeHSMM(x, M = NA, HSMM, obsdist, dwelldist)
}
\arguments{
\item{x}{Numeric vector. The observed data sequence.}

\item{M}{Integer. Maximum dwell time to consider for semi-Markov states. Defaults to \code{min(length(x), 1000)} if \code{NA}.}

\item{HSMM}{A list containing estimated HSMM parameters, including \code{observationparameters}, \code{dwellparameters}, \code{Pi}, and \code{delta}. Typically returned by \code{\link{findmleHSMM}}.}

\item{obsdist}{Character string. Observation distribution. Supported distributions: "norm", "pois", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}

\item{dwelldist}{Character string. Dwell-time distribution. Supported distributions: "pois", "nbinom", "betabinom".}

}
\value{
\item{states}{Numeric vector. The most likely hidden state at each observation, computed via local decoding.}
}
\details{
This function uses the forward-backward algorithm for HSMMs to compute local
state probabilities and returns the most probable state at each time point.
It supports a wide range of observation distributions and dwell-time distributions.
The user can supply a maximum dwell-time truncation parameter \code{M} and
optionally use a shift parameter for dwell times.
}
\examples{
J <- 3
Pi <- matrix(c(0.0, 0.6, 0.4,
               0.5, 0.0, 0.5,
               0.3, 0.7, 0.0), nrow = J, byrow = TRUE)

obspar <- list(
  mean = c(-4, 0, 4),
  sd   = c(1, 0.5, 1.5)
)

dwellpar <- list(
  lambda = c(3, 5, 4)
)

# Simulate HSMM data
sim <- generateHSMM(n = 1000, J = J, obsdist = "norm",
                    dwelldist = "pois", obspar = obspar,
                    dwellpar = dwellpar, Pi = Pi)

# Fit HSMM using the true parameters
HSMM_true <- findmleHSMM(x = sim$x, J = J, M = 100,
                          obsdist = "norm", dwelldist = "pois",
                          obspar = obspar, dwellpar = dwellpar,
                          Pi = Pi)

# Decode states using localdecodeHSMM
decoded <- localdecodeHSMM(x = sim$x, HSMM = HSMM_true,
                            obsdist = "norm", dwelldist = "pois",
                            M = 100)
decoded

}
\seealso{
\code{\link{findmleHSMM}}, for estimating HSMM parameters from data.
\code{\link{generateHSMM}}, for simulating HSMM data.
\code{\link{localdecodeHMM}}, for local decoding of a hidden Markov model.
\code{\link{globaldecodeHSMM}}, for finding the most likely sequence of states using global decoding.
}
\author{Aimee Cody}

