generateHSMM <- function(n, J, obsdist, dwelldist, obspar, dwellpar, Pi, delta = NULL, simtype = "nobs", shift = FALSE, seed = NULL) {

  # Set seed if provided
  if (!is.null(seed)) {
    set.seed(seed)
  }

  # Calculate stationary distribution if initial probabilities not provided
  if (is.null(delta)) {
    delta <- solve(t(diag(J) - Pi + 1), rep(1, J))
  }

  # Validate simulation type
  if (!(simtype %in% c("nseq", "nobs"))) {
    stop("simtype must be either 'nseq' or 'nobs'")
  }

  # Set default shift parameters if not using provided shifts
  if (!shift) {
    if (is.null(dwellpar$shift)) {
      dwellpar$shift <- rep(1, J)
    }
  }

  # Generate state sequence using transition probabilities
  simulate_states <- function(delta, Pi, n) {
    states <- numeric(n)
    states[1] <- sample(1:length(delta), 1, prob = delta)
    for (t in 2:n) {
      states[t] <- sample(1:ncol(Pi), 1, prob = Pi[states[t - 1], ])
    }
    return(states)
  }

  # Generate dwell time for a given state
  generate_dwell <- function(state) {
    # Validate state index
    if (state < 1 || state > J) {
      stop(paste0("State index out of bounds: state = ", state, ", expected between 1 and ", J))
    }

    # Generate dwell time based on specified distribution
    if (dwelldist == "pois") {
      # Shifted Poisson distribution
      if (is.null(dwellpar$lambda) || is.null(dwellpar$shift)) {
        stop("Please supply 'lambda' and 'shift' for dwell parameters")
      }
      val <- rpois(1, dwellpar$lambda[state]) + dwellpar$shift[state]

    } else if (dwelldist == "nbinom") {
      # Shifted negative binomial distribution
      if (is.null(dwellpar$shift) || is.null(dwellpar$size) || is.null(dwellpar$mu)) {
        stop("Please supply 'shift', 'size' and 'mu' for dwell parameters")
      }
      val <- rnbinom(1, size = dwellpar$size[state], mu = dwellpar$mu[state]) + dwellpar$shift[state]

    } else if (dwelldist == "betabinom") {
      # Shifted beta-binomial distribution
      if (is.null(dwellpar$size) || is.null(dwellpar$alpha) || is.null(dwellpar$beta) || is.null(dwellpar$shift)) {
        stop("Please supply 'size', 'alpha', 'beta' and 'shift' for dwell parameters")
      }
      val <- rbetabinom(1, size = dwellpar$size[state], alpha = dwellpar$alpha[state], beta = dwellpar$beta[state]) + dwellpar$shift[state]

    } else {
      stop("Unsupported sojourn distribution type")
    }

    # Validate generated dwell time
    if (is.na(val) || val < 1 || !is.finite(val)) {
      stop(paste0("Invalid dwell time generated for state ", state, ": ", val))
    }

    return(floor(val))
  }

  # Generate initial state sequence based on simulation type
  if (simtype == "nseq") {
    # Generate exactly n state transitions
    s0 <- simulate_states(delta, Pi, n)
  } else {
    # Generate enough states to reach n observations
    total_length <- 0
    s0 <- numeric(0)
    while (total_length < n) {
      new_seq <- simulate_states(delta, Pi, ceiling(n / 5))
      u_temp <- sapply(new_seq, generate_dwell)
      if (any(is.na(u_temp))) stop("NA encountered in dwell time generation")
      if (any(u_temp < 1)) stop("Dwell times must all be >= 1")
      s0 <- c(s0, new_seq)
      total_length <- total_length + sum(u_temp)
    }
  }

  # Generate dwell times for state sequence
  u <- sapply(s0, generate_dwell)
  if (any(is.na(u))) stop("NA encountered in dwell time generation")
  if (any(u < 1)) stop("Dwell times must all be >= 1")

  # Expand states according to their dwell times
  s1 <- rep(s0, u)

  # Adjust sequence length for observation-based simulation
  if (simtype == "nobs") {
    if (length(s1) > n) {
      s1 <- s1[1:n]
    } else if (length(s1) < n) {
      while (length(s1) < n) {
        extra_states <- simulate_states(delta, Pi, max(10, ceiling((n - length(s1)) / 3)))
        extra_u <- sapply(extra_states, generate_dwell)
        if (any(is.na(extra_u))) stop("NA encountered in dwell time generation (extra)")
        if (any(extra_u < 1)) stop("Dwell times must all be >= 1 (extra)")
        s1 <- c(s1, rep(extra_states, extra_u))
      }
      s1 <- s1[1:n]
    }
  }

  # Generate observations for each time point
  x <- numeric(length(s1))
  for (i in seq_along(s1)) {
    if (obsdist == "norm") {
      # Normal distribution
      if (is.null(obspar$mean) || is.null(obspar$sd)) {
        stop("Please supply 'mean' and 'sd' for observation parameters")
      }
      x[i] <- rnorm(1, obspar$mean[s1[i]], obspar$sd[s1[i]])

    } else if (obsdist == "pois") {
      # Poisson distribution
      if (is.null(obspar$lambda)) {
        stop("Please supply 'lambda' for observation parameters")
      }
      x[i] <- rpois(1, obspar$lambda[s1[i]])

    } else if (obsdist == "weibull") {
      # Weibull distribution
      if (is.null(obspar$shape) || is.null(obspar$scale)) {
        stop("Please supply 'shape' and 'scale' for observation parameters")
      }
      x[i] <- rweibull(1, shape = obspar$shape[s1[i]], scale = obspar$scale[s1[i]])

    } else if (obsdist == "zip") {
      # Zero-inflated Poisson distribution
      if (is.null(obspar$pi) || is.null(obspar$lambda)) {
        stop("Please supply 'pi' and 'lambda' for observation parameters")
      }
      if (runif(1) < obspar$pi[s1[i]]) {
        x[i] <- 0  # Zero with probability pi
      } else {
        x[i] <- rpois(1, lambda = obspar$lambda[s1[i]])  # Poisson otherwise
      }

    } else if (obsdist == "nbinom") {
      # Negative binomial distribution
      if (is.null(obspar$size) || is.null(obspar$mu)) {
        stop("Please supply 'size' and 'mu' for observation parameters")
      }
      x[i] <- rnbinom(1, size = obspar$size[s1[i]], mu = obspar$mu[s1[i]])

    } else if (obsdist == "zinb") {
      # Zero-inflated negative binomial distribution
      if (is.null(obspar$pi) || is.null(obspar$size) || is.null(obspar$mu)) {
        stop("Please supply 'pi', 'size' and 'mu' for observation parameters")
      }
      if (runif(1) < obspar$pi[s1[i]]) {
        x[i] <- 0  # Zero with probability pi
      } else {
        x[i] <- rnbinom(1, size = obspar$size[s1[i]], mu = obspar$mu[s1[i]])
      }

    } else if (obsdist == "exp") {
      # Exponential distribution
      if (is.null(obspar$rate)) {
        stop("Please supply 'rate' for observation parameters")
      }
      x[i] <- rexp(1, rate = obspar$rate[s1[i]])

    } else if (obsdist == "gamma") {
      # Gamma distribution
      if (is.null(obspar$shape) || is.null(obspar$rate)) {
        stop("Please supply 'shape' and 'rate' for observation parameters")
      }
      x[i] <- rgamma(1, shape = obspar$shape[s1[i]], rate = obspar$rate[s1[i]])

    } else if (obsdist == "lnorm") {
      # Log-normal distribution
      if (is.null(obspar$meanlog) || is.null(obspar$sdlog)) {
        stop("Please supply 'meanlog' and 'sdlog' for observation parameters")
      }
      x[i] <- rlnorm(1, meanlog = obspar$meanlog[s1[i]], sdlog = obspar$sdlog[s1[i]])

    } else if (obsdist == "gev") {
      # Generalized extreme value distribution
      if (is.null(obspar$loc) || is.null(obspar$scale) || is.null(obspar$shape)) {
        stop("Please supply 'loc', 'scale' and 'shape' for observation parameters")
      }
      # Validate GEV parameters
      loc_i <- obspar$loc[s1[i]]
      scale_i <- obspar$scale[s1[i]]
      shape_i <- obspar$shape[s1[i]]

      if (is.na(scale_i) || scale_i <= 0 || is.na(loc_i) || is.na(shape_i)) {
        stop(sprintf("Invalid GEV parameter for state %d: loc=%s, scale=%s, shape=%s",
                     s1[i], loc_i, scale_i, shape_i))
      }
      x[i] <- revd(1, loc = obspar$loc[s1[i]], scale = obspar$scale[s1[i]], shape = obspar$shape[s1[i]], type = "GEV")

    } else if (obsdist == "ZInormal") {
      # Zero-inflated normal distribution
      if (is.null(obspar$pi) || is.null(obspar$mean) || is.null(obspar$sd)) {
        stop("Please supply 'pi', ''mean' and 'sd' for observation parameters")
      }
      if (runif(1) < obspar$pi[s1[i]]) {
        x[i] <- 0  # Zero with probability pi
      } else {
        x[i] <- rnorm(1, mean = obspar$mean[s1[i]], sd = obspar$sd[s1[i]])
      }

    } else if (obsdist == "ZIgamma") {
      # Zero-inflated negative gamma distribution
      if (is.null(obspar$pi) || is.null(obspar$shape) || is.null(obspar$rate)) {
        stop("Please supply 'pi', 'shape' and 'rate' for observation parameters")
      }
      if (runif(1) < obspar$pi[s1[i]]) {
        x[i] <- 0  # Zero with probability pi
      } else {
        x[i] <- rgamma(1, shape = obspar$shape[s1[i]], rate = obspar$rate[s1[i]])
      }

    } else {
      stop("Observation distribution not supported")
    }
  }

  # Return results
  result <- list(
    states = s1,
    x = x,
    N = length(x)
  )
  return(result)
}











