#' Plot the U-Matrix of a Gower-SOM
#'
#' Visualizes the U-Matrix of a trained Gower-SOM using `ggplot2`. The U-Matrix
#' encodes average distances between each neuron and its immediate neighbors,
#' and can be used to reveal cluster boundaries and topological structures.
#'
#' @param u_matrix A numeric matrix as returned by \code{\link{gsom_Umatrix}},
#'   with dimensions \code{n_rows x n_cols}.
#' @param fill_palette Character, palette to use for the fill scale. Defaults to
#'   `"C"` (viridis option).
#'
#' @return A `ggplot` object displaying the U-Matrix.
#'
#' @details
#' The function uses `geom_raster()` to display the U-Matrix as a heatmap.
#' Viridis palettes are used by default for perceptual uniformity, but this can
#' be changed with the argument `fill_palette`.
#'
#' @examples
#' \dontrun{
#' set.seed(1)
#' df <- data.frame(
#'   x1 = rnorm(20),
#'   x2 = rnorm(20),
#'   g  = factor(sample(letters[1:3], 20, TRUE))
#' )
#' fit <- gsom_Training(df, grid_rows = 3, grid_cols = 3,
#'                 num_iterations = 5, batch_size = 4)
#' U <- gsom_Umatrix(fit$weights, n_rows = 3, n_cols = 3)
#' plot_Umatrix(U)
#' }
#'
#' @seealso \code{\link{gsom_Umatrix}}
#' @export
#' @importFrom ggplot2 ggplot aes geom_raster theme_void scale_fill_viridis_c theme element_blank element_text
#' @importFrom reshape2 melt
#' @importFrom grid unit

plot_Umatrix <- function(u_matrix, fill_palette = "C") {
  longData <- reshape2::melt(u_matrix)
  Var1 <- colnames(longData)[1]
  Var2 <- colnames(longData)[2]
  value <- colnames(longData)[3]
  ggplot2::ggplot(data = longData, ggplot2::aes(x = Var2, y = Var1, fill = value)) +
    ggplot2::geom_raster() +
    ggplot2::labs(title = "Gower-SOM U-Matrix")+
    ggplot2::scale_fill_viridis_c(option = fill_palette) +
    ggplot2::theme_void() +
    ggplot2::theme(
      axis.title.x = ggplot2::element_blank(),
      axis.text.x  = ggplot2::element_blank(),
      axis.title.y = ggplot2::element_blank(),
      axis.text.y  = ggplot2::element_blank(),
      axis.ticks.x = ggplot2::element_blank(),
      axis.line    = ggplot2::element_blank(),
      axis.text    = ggplot2::element_blank(),
      axis.ticks   = ggplot2::element_blank(),
      axis.title   = ggplot2::element_blank(),
      panel.grid.major = ggplot2::element_blank(),
      panel.grid.minor = ggplot2::element_blank(),
      legend.text  = ggplot2::element_text(size = 14),
      legend.title = ggplot2::element_blank(),
      legend.position = "bottom",
      legend.key.size = grid::unit(1.2, "cm")
    )
}
