% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GALAHAD.R
\name{GALAHAD}
\alias{GALAHAD}
\title{GALAHAD: Geometry-Adaptive Lyapunov-Assured Hybrid Optimizer}
\usage{
GALAHAD(V, gradV, theta0, parts, control = list(), callback = NULL)
}
\arguments{
\item{V}{Objective function: \code{function(theta) -> scalar}}

\item{gradV}{Gradient function: \code{function(theta) -> vector of length p}}

\item{theta0}{Initial parameter vector (numeric, length p)}

\item{parts}{List with geometry partitions: \code{T}, \code{P}, \code{E}.
See Details.}

\item{control}{Optional list of control parameters. See Details.}

\item{callback}{Optional progress callback:
\code{function(info)} where info has \code{iter}, \code{theta},
\code{value}, \code{grad_norm}}
}
\value{
List with components:
\describe{
\item{\code{theta}}{Final parameter vector}
\item{\code{value}}{Final objective value}
\item{\code{grad_inf}}{Infinity norm of final gradient}
\item{\code{converged}}{Logical convergence flag}
\item{\code{status}}{Convergence status string}
\item{\code{reason}}{Detailed convergence reason}
\item{\code{iterations}}{Number of iterations performed}
\item{\code{history}}{data.frame with iteration history}
\item{\code{diagnostics}}{List with convergence diagnostics and
Lyapunov certificates}
\item{\code{certificate}}{Convergence certificate}
}
}
\description{
Battle-hardened production optimizer with geometry awareness,
Lyapunov stability monitoring, and trust-region safety.
}
\details{
\strong{Geometry Partitions:}

Parameters are divided into three geometric types:
\describe{
\item{\code{T} (log-scale)}{Natural gradient on positive reals.
Use for scale parameters spanning orders of magnitude
(e.g., \eqn{\sigma \in (0.01, 100)}).}
\item{\code{P} (positive orthant)}{Entropy mirror descent.
Use for positive parameters with moderate range
(e.g., \eqn{\alpha \in (0.1, 10)}).}
\item{\code{E} (Euclidean)}{Standard gradient descent.
Use for unconstrained parameters (e.g., regression coefficients).}
}

\strong{Control Parameters:}
\describe{
\item{\code{max_iter}}{Maximum iterations (default: 2000)}
\item{\code{tol_g}}{Gradient tolerance (default: 1e-6)}
\item{\code{tol_x}}{Step tolerance (default: 1e-9)}
\item{\code{tol_f}}{Function change tolerance (default: 1e-12)}
\item{\code{delta}}{Initial trust radius (default: 1.0)}
\item{\code{eta0}}{Initial step size (default: 1.0)}
\item{\code{V_star}}{Known minimum (optional, for Polyak steps)}
\item{\code{lambda}}{L2 regularization weight (default: 0)}
}
}
\examples{
\donttest{
# Quadratic objective
p <- 20
Q <- diag(1:p)
theta_true <- rnorm(p)
V <- function(th) 0.5 * sum((th - theta_true) * (Q \%*\% (th - theta_true)))
gradV <- function(th) Q \%*\% (th - theta_true)

# Mixed geometry: log-scale, positive, Euclidean
parts <- list(T = 1:5, P = 6:10, E = 11:20)
theta0 <- abs(rnorm(p)) + 0.1

# Set seed for reproducibility (outside the function)
set.seed(42)

# Optimize with progress tracking
result <- GALAHAD(V, gradV, theta0, parts,
  control = list(max_iter = 100, tol_g = 1e-6),
  callback = function(info) {
    if (info$iter \%\% 10 == 0) {
      cat(sprintf("Iter \%3d: V = \%.6f, ||g|| = \%.3e\n",
                  info$iter, info$value, info$grad_norm))
    }
  })

print(result$theta)
print(result$diagnostics)
}

}
\references{
Conn, A. R., Gould, N. I., & Toint, P. L. (2000).
\emph{Trust-region methods}. SIAM.

Amari, S. (1998). Natural gradient works efficiently in learning.
\emph{Neural computation}, 10(2), 251-276.
}
