
#' @title Check metric properties of cost matrices
#' @description Check if a cost matrix satisfies symmetry, positive definiteness and the triangle inequality.
#' @param x numeric square matrix.
#' @param tol.sym tolerance used to check symmetry.
#' @param tol.pd tolerance used to check positive definiteness.
#' @param tol.ti tolerance used to check the triangle inequality.
#' @details The following three properties of a square matrix \eqn{x \in \mathbb{R}^{N \times N}}
#' are checked:
#' \itemize{
#'   \item symmetry; if \eqn{x_{ij} = x_{ji}},
#'   \item positive definiteness; if \eqn{x_{ii} = 0} and \eqn{x_{ij} > 0} for all \eqn{i \neq j},
#'   \item triangle inequality; if \eqn{x_{ij} \leq x_{ik} + x_{kj}}.
#' }
#' If symmetry and positive definiteness are satisfied, then \eqn{x} is called a semi-metric
#' cost matrix. If additionally also the triangle inequality holds, then \eqn{x} is a
#' metric cost matrix.
#' @returns A list containing logical entries `metric`, `semi.metric`, `sym`, `pos.def`
#' and `tri.ineq` that indicate whether the corresponding property is satisfied.
#' @examples
#' x <- cost_matrix_lp(1:5)
#' res <- is_metric_cost_mat(x)
#' res2 <- is_metric_cost_mat(x^2)
#' # x is a metric cost matrix
#' print(res$metric)
#' # x^2 is only a semi-metric cost matrix,
#' # because the triangle inequality is not satisfied
#' print(res2$semi.metric)
#' print(res2$tri.ineq)
#' @seealso [`cost_matrix_lp`]
#' @export
is_metric_cost_mat <- \(x, tol.sym = 1e-8, tol.pd = 0, tol.ti = 1e-8) {
    stopifnot(
        is_num_mat(x),
        is_nonneg_scalar(tol.sym),
        is_nonneg_scalar(tol.pd),
        is_nonneg_scalar(tol.ti)
    )

    N <- nrow(x)
    stopifnot(dim(x) == c(N, N))

    sym <- if (tol.sym == Inf) TRUE else isSymmetric.matrix(x, tol = tol.sym)
    pd <- if (tol.pd == Inf) TRUE else all(x >= -tol.pd) &&
        isTRUE(all.equal(diag(x), rep(0, N), tolerance = tol.pd, check.attributes = FALSE))
    ti <- if (tol.ti == Inf) TRUE else satisfiesTriangleInequality(x, tol.ti)
    sm <- sym && pd
    me <- sm && ti

    list(
        metric      = me,
        semi.metric = sm,
        sym         = sym,
        pos.def     = pd,
        tri.ineq    = ti
    )
}
