#' plotSegs
#'
#' This function is used to create a plot of a samples segs. The input samples
#' can either be a single data.frame of the segs of one patient or a list of
#' data.frames for multiple samples.
#'
#' @param samples The samples to plot. If a list it plots both on the same plot
#' @param name The name of the sample. Used for plot title
#' @param chrom Which chromosome to plot. Default plots all of them.
#' @param gbuild The reference genome build. Default is hg19. Also supports hg18 and hg38.
#' @param sep Whether or not to place different members of the list on the same
#'            or different axis
#' @param alpha Allows you to adjust the transparency of the lines. 0-1
#' @keywords plot segments
#' @return displays a plot of the segments
#' @import ggplot2
#' @export
#' @examples
#' plotSegs(segDataExp[[1]]) #Plots all of the first sample's segments
#' plotSegs(segDataExp[[1]],1) #Only plots the first chromosome segments
#' plotSegs(segDataExp[1:2]) #Plots first two samples on same axis
#' plotSegs(segDataExp[1:2], sep = TRUE) #Plots first two samples seperately
plotSegs = function(samples, name="", chrom = -1,gbuild = "hg19", sep = FALSE, alpha = 1) {

   #Set up restore for par
   oldpar <- par(no.readonly = TRUE)
   on.exit(par(oldpar))

   #Get chromosome lengths from built in datasets
   if (gbuild == "hg19") {
      chrlen = hg19.chrom.sizes[1:24,]
   } else if (gbuild == "hg18") {
      chrlen = hg18.chrom.sizes[1:24,]
   } else if (gbuild == "hg38"){
      chrlen = hg38.chrom.sizes[1:24,]
   } else {
      msg = "Unrecognized genome build. Check Parameters."
      stop(msg)
   }
   
   #Formats the chrlen data
   chrlen[,1] = sapply(chrlen[,1],function(x){substr(x,4,nchar(x))})
   chrlen[chrlen[,1]=="X",1] = 23
   chrlen = chrlen[chrlen[,1]!="Y",]
   
   #Gets the max coordinate for each chromosome
   maxChrom = chrlen[order(as.numeric(chrlen[,1])),2]
   maxChrom = c(0,maxChrom[1:22])

   if (chrom > 0){
      coords<-c(sum(maxChrom[1:chrom]),sum(maxChrom[1:(chrom+1)]))
   } else{
      coords<-c(0,sum(chrlen[,2]))
   }
   
   # Custom function to convert a position and chromosome and to a genome
   # wide position
   chrPos = function(pos,chr) {return(sum(maxChrom[c(1:chr)])+pos)}
   
   # Finds the number of samples to plot
   if (is.data.frame(samples))
      numSamples = 1
   else if (is.list(samples))
      numSamples = length(samples)
   
   # Gets the color using the specified alpha value
   aVal = alpha
   colors = c(alpha("blue",aVal),alpha("red",aVal),alpha("purple",aVal),
              alpha("brown",aVal),alpha("green",aVal))
   
   # Changes the number of rows if the plots are to be seperated
   if (sep){
      graphics::par(mfrow=c(numSamples,1))
   }
   else{
      plot(1, type="n", xlab="Genome Position", ylab="SegVal", main=name, 
           xlim=coords, ylim=c(0,10))
   }
   
   # Goes through each of the samples and adds the lines to the plots
   for (j in 1:numSamples){
      if (is.data.frame(samples))
         samp = samples
      else
         samp = samples[[j]]
      samp$chromosome[samp$chromosome=="X"]<-23
      
      # If it should be seperated it generates a new plot
      if (sep){
         plot(1, type="n", xlab="Genome Position", ylab="SegVal", 
              main=paste0(names(samples)[j], " (",nrow(samp),")"), xlim=coords, 
              ylim=c(0,10))
      }
      
      # Skips sample if it has no segments
      if(nrow(samp)==0)
         next()
      
      # Plots all of the segments as a line
      for (i in 1:nrow(samp)){
         graphics::lines(c(chrPos(samp[i,2],samp[i,1]),chrPos(samp[i,3],samp[i,1])),
               c(samp[i,4],samp[i,4]),col=colors[j])
      }
   }
   
   # If the samples are plotted on the same plot it adds a legend
   if (!is.data.frame(samples) & !sep)
      graphics::legend(coords[1], 10, legend=names(samples), 
                       col=colors[c(1:length(samples))],lty=1, cex=0.8)
}
