\name{modwt}
\alias{modwt}
\alias{imodwt}
\title{(Inverse) Maximal Overlap Discrete Wavelet Transform}
\description{
  This function performs a level \eqn{J} decomposition of the input
  vector using the non-decimated discrete wavelet transform. The inverse
  transform performs the reconstruction of a vector or time series from
  its maximal overlap discrete wavelet transform.
}
\usage{modwt(x, wf = "la8", n.levels = 4, boundary = "periodic")
imodwt(y)
}
\arguments{
  \item{x}{
    a vector or time series containing the data be to decomposed.  There
    is \bold{no} restriction on its length.
  }
  \item{y}{
    Object of class \code{"modwt"}.
  }
  \item{wf}{
    Name of the wavelet filter to use in the decomposition.  By default
    this is set to \code{"la8"}, the Daubechies orthonormal compactly
    supported wavelet of length L=8 (Daubechies, 1992), least
    asymmetric family. 
  }
  \item{n.levels}{
    Specifies the depth of the decomposition.  This must be a number
    less than or equal to log(length(x),2).
  }
  \item{boundary}{
    Character string specifying the boundary condition.   If
    \code{boundary=="periodic"} the defaulTRUE, then the vector you
    decompose is assumed to be periodic on its defined interval,\cr
    if \code{boundary=="reflection"}, the vector beyond its boundaries
    is assumed to be a symmetric reflection of itself.
  }
}
\value{
  Object of class \code{"modwt"}, basically, a list with the following
  components 
  \item{d?}{Wavelet coefficient vectors.}
  \item{s?}{Scaling coefficient vector.}
  \item{wavelet}{Name of the wavelet filter used.}
  \item{boundary}{How the boundaries were handled.}
}
\details{
  The code implements the one-dimensional non-decimated DWT using the
  pyramid algorithm.  The actual transform is performed in C using
  pseudocode from Percival and Walden (2001).  That means convolutions,
  not inner products, are used to apply the wavelet filters.

  The MODWT goes by several names in the statistical and engineering
  literature, such as, the ``stationary DWT'', ``translation-invariant
  DWT'', and ``time-invariant DWT''.

  The inverse MODWT implements the one-dimensional inverse transform
  using the pyramid algorithm (Mallat, 1989). 
}
\references{
  Gencay, R., F. Selcuk and B. Whitcher (2001)
  \emph{An Introduction to Wavelets and Other Filtering Methods in
    Finance and Economics},
  Academic Press.
  
  Percival, D. B. and P. Guttorp (1994)
  Long-memory processes, the Allan variance and wavelets,
  In \emph{Wavelets and Geophysics}, pages 325-344, Academic Press.

  Percival, D. B. and A. T. Walden (2000)
  \emph{Wavelet Methods for Time Series Analysis},
  Cambridge University Press.
}
\seealso{
  \code{\link{dwt}}, \code{\link{idwt}}, \code{\link{mra}}.
}
\examples{
## Figure 4.23 in Gencay, Selcuk and Whitcher (2001)
data(ibm)     
ibm.returns <- diff(log(ibm))
# Haar
ibmr.haar <- modwt(ibm.returns, "haar")
names(ibmr.haar) <- c("w1", "w2", "w3", "w4", "v4")
# LA(8)
ibmr.la8 <- modwt(ibm.returns, "la8")
names(ibmr.la8) <- c("w1", "w2", "w3", "w4", "v4")
# shift the MODWT vectors
ibmr.la8 <- phase.shift(ibmr.la8, "la8")
## plot partial MODWT for IBM data
par(mfcol=c(6,1), pty="m", mar=c(5-2,4,4-2,2))
plot.ts(ibm.returns, axes=FALSE, ylab="", main="(a)")
for(i in 1:5)
  plot.ts(ibmr.haar[[i]], axes=FALSE, ylab=names(ibmr.haar)[i])
axis(side=1, at=seq(0,368,by=23), 
  labels=c(0,"",46,"",92,"",138,"",184,"",230,"",276,"",322,"",368))
par(mfcol=c(6,1), pty="m", mar=c(5-2,4,4-2,2))
plot.ts(ibm.returns, axes=FALSE, ylab="", main="(b)")
for(i in 1:5)
  plot.ts(ibmr.la8[[i]], axes=FALSE, ylab=names(ibmr.la8)[i])
axis(side=1, at=seq(0,368,by=23), 
  labels=c(0,"",46,"",92,"",138,"",184,"",230,"",276,"",322,"",368))
}
\author{B. Whitcher}
\keyword{ts}
