% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/phylo_spectro.R
\name{phylo_spectro}
\alias{phylo_spectro}
\title{Add spectrograms onto phylogenetic trees}
\usage{
phylo_spectro(X, tree, type = "phylogram", par.mar = rep(1, 4), 
size = 1, offset = 0, path = NULL, ladder = NULL, horizontal = TRUE, ...)
}
\arguments{
\item{X}{'selection_table', 'extended_selection_table' or data frame containing columns for sound file name 
(sound.files), selection number (selec), and start and end time of signals (start and end). 
'top.freq' and 'bottom.freq' columns are optional. In addition, the data frame must include the column 'tip.label' that contains the names of the tip labels found in the tree (e.g. '\code{tree$tip.label}). This column is used to match rows and tip labels. If using an 
'extended_selection_table' the sound files are not required (see \code{\link{selection_table}}).}

\item{tree}{Object of class 'phylo' (i.e. a phylogenetic tree). Ultrametric trees may produce better results.
If \code{NULL} (default) then the current working directory is used. Tip labels must match the names provided in the 'tip.label' column in 'X' (see 'X' argument).}

\item{type}{Character string of length 1 specifying the type of phylogeny to be drawn 
(as in \code{\link[ape]{plot.phylo}}). Only 'phylogram' (default) and 'fan' are allowed.}

\item{par.mar}{Numeric vector with 4 elements, default is \code{rep(1, 4)}. Specifies the number of lines 
in inner plot margins where axis labels fall, with form c(bottom, left, top, right). 
See \code{\link[graphics]{par}}. See 'inner.par' argument for controlling spectrogram margins.}

\item{size}{Numeric vector of length 1 controlling the relative size of spectrograms. Higher numbers increase the height of spectrograms. Default is 1. 
Numbers between range \code{c(>0, Inf)} are allowed.}

\item{offset}{Numeric vector of length 1 controlling the space between tips and spectrograms. Default is 0.}

\item{path}{Character string containing the directory path where the sound files are located. 
If \code{NULL} (default) then the current working directory is used.}

\item{ladder}{Character string controlling whether the phylogeny is ladderized (i.e. the internal structure of the 
tree is reorganized to get the ladderized effect when plotted). Only 'left' of 'right' values are accepted. Default is 
\code{NULL} (no ladderization). See \code{\link[ape]{ladderize}} for more details.}

\item{horizontal}{Logical. Controls whether spectrograms in a fan phylogeny are place in a horizontal position 
\code{FALSE} or in the same angle as the tree tips. Currently only horizontal spectrograms are available.}

\item{...}{Additional arguments to be passed to the internal spectrogram 
creating function (\code{\link{specreator}}) or phylogeny plotting function (\code{\link[ape]{plot.phylo}}) for 
customizing graphical output. Only rightwards phylogenies can be plotted.}
}
\value{
A phylogenetic tree with spectrograms on tree tips is plotted in the current graphical device.
}
\description{
\code{phylo_spectro} Add spectrograms to the tips of an objects of class phylo.
}
\details{
The function add the spectrograms of sounds annotated in a selection table ('X' argument) onto the tips of a phylogenetic tree. 
The 'tip.label' column in 'X' is used to match spectrograms and tree tips. The function uses internally the \code{\link[ape]{plot.phylo}} function to plot the tree 
and the \code{\link{specreator}} function to create the spectrograms. Arguments for both of these functions
can be provided for further customization.
}
\examples{
{ 
\donttest{
# First set empty folder


# save example sound files
data(list = c("Phae.long1", "Phae.long2", "Phae.long3", "lbh_selec_table"))
writeWave(Phae.long1, file.path(tempdir(), "Phae.long1.wav"))
writeWave(Phae.long2, file.path(tempdir(), "Phae.long2.wav"))
writeWave(Phae.long3, file.path(tempdir(), "Phae.long3.wav"))

# set spectrogram options (can be done at the phylo_spectro() function too)
warbleR_options(wl = 200, ovlp = 90, flim = "frange", wav.path = tempdir())

# subset example selection table
X <- lbh_selec_table[1:8, ]

# create random tree (need ape to be installed)
set.seed(1)
tree <- ape::rtree(nrow(X))

# Force tree to be ultrametric
tree <- ape::chronoMPL(tree)

# add tip label column to example selection table (just for the sake of the example)
X$tip.label <- tree$tip.label

# print phylogram with spectros
phylo_spectro(X = X, tree = tree, par.mar = c(0, 0, 0, 8), size = 2)

# no margin in spectrograms and showing tip labels (higher offset)
phylo_spectro(X = X, tree = tree, offset = 0.1, par.mar = c(0, 0, 0, 6), 
inner.mar = rep(0, 4), size = 2)

# print fan tree and no margin in spectrograms
phylo_spectro(X = X, tree = tree, offset = 0.6, par.mar = rep(3, 4),
inner.mar = rep(0, 4), size = 2, type = "fan", show.tip.label = FALSE)

# changing edge color and witdh
phylo_spectro(X = X, tree = tree, offset = 0.2, par.mar = rep(3, 4), inner.mar = rep(0, 4), 
size = 2, type = "fan", show.tip.label = FALSE, edge.color = "red", edge.width = 2)

# plotting a tree representing cross-correlation distances 
xcorr_mat <- xcorr(X, bp = c(1, 10))

xc.tree <- ape::chronoMPL(ape::as.phylo(hclust(as.dist(1 - xcorr_mat))))

X$tip.label <- xc.tree$tip.label

phylo_spectro(X = X, tree = xc.tree, offset = 0.03, par.mar = rep(3, 4), 
inner.mar = rep(0, 4), size = 0.3, type = "fan", show.tip.label = FALSE, 
edge.color = "red", edge.width = 2)
  }
}
}
\references{
{
Araya-Salas, M., & Smith-Vidaurre, G. (2017). warbleR: An R package to streamline analysis of animal acoustic signals. Methods in Ecology and Evolution, 8(2), 184-191.
}
}
\seealso{
\code{\link{specreator}}, \code{\link[ape]{plot.phylo}}

Other spectrogram creators: \code{\link{color.spectro}},
  \code{\link{dfDTW}}, \code{\link{dfts}},
  \code{\link{ffDTW}}, \code{\link{ffts}},
  \code{\link{multi_DTW}}, \code{\link{snrspecs}},
  \code{\link{sp.en.ts}}, \code{\link{specreator}},
  \code{\link{trackfreqs}}
}
\author{
Marcelo Araya-Salas (\email{marceloa27@gmail.com})
}
\concept{spectrogram creators}
