% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BackgroundSamplingFunctions.R
\name{mSampling3D}
\alias{mSampling3D}
\title{3D background sampling}
\usage{
mSampling3D(occs, envBrick, mShp, depthLimit = "all")
}
\arguments{
\item{occs}{A \code{data.frame} with at least three columns
named "longitude", "latitude", and "depth", or that
can be coerced into this format.}

\item{envBrick}{A \code{RasterBrick} object to serve
as a template for generating background sampling
coordinates.}

\item{mShp}{A shapefile defining the area from
which background points should be sampled.}

\item{depthLimit}{An argument controlling the depth
extent of sampling. Refer to \code{Details} for more information.}
}
\value{
A \code{data.frame} with 3D coordinates of points for background
sampling.
}
\description{
Samples XYZ coordinates from a shapefile
from maximum to minimum occurrence depth at XYZ
resolution of envBrick.
}
\details{
This function is designed to sample background points for
distributional modeling in three dimensions. If a voxel (3D pixel)
in the \code{envBrick} intersects with an occurrence from \code{occs}, it is
removed. Note that this function returns points representing every
voxel in the background area within the specified depth range. It
is up to the user to downsample from these data as necessary,
depending on the model type being used.

\code{depthLimit} argument options:
\itemize{
\item \code{occs} Samples background from the full depth extent of \code{occs}.
\item \code{all} Samples background from the full depth extent of \code{envBrick}.
\item A \code{vector} of length 2 with maximum and minimum depth values from
which to sample.
}
}
\examples{
library(raster)

# Create test raster
r1 <- raster(ncol=10, nrow=10)
values(r1) <- 1:100
r2 <- raster(ncol=10, nrow=10)
values(r2) <- c(rep(20, times = 50), rep(60, times = 50))
r3 <- raster(ncol=10, nrow=10)
values(r3) <- 8
envBrick <- brick(r1, r2, r3)
names(envBrick) <- c(0, 10, 30)

# Create test occurrences
set.seed(0)
longitude <- sample(extent(envBrick)[1]:extent(envBrick)[2],
                    size = 10, replace = FALSE)
set.seed(0)
latitude <- sample(extent(envBrick)[3]:extent(envBrick)[4],
                   size = 10, replace = FALSE)
set.seed(0)
depth <- sample(0:35, size = 10, replace = TRUE)
occurrences <- as.data.frame(cbind(longitude,latitude,depth))

# Generate background sampling buffer
buffPts <- SpatialPoints(occurrences[,c("longitude", "latitude")])
crs(buffPts) <- "+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs"
mShp <- buffer(buffPts, width = 1, dissolve = TRUE)

# Here's the function
occSample3d <- mSampling3D(occurrences,
                           envBrick, mShp = mShp,
                           depthLimit = "occs")

}
\keyword{backgroundSampling}
