% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eocusum_sim.R
\name{optimal_k}
\alias{optimal_k}
\title{Compute approximate optimal k}
\usage{
optimal_k(pmix, RA, yemp = FALSE)
}
\arguments{
\item{pmix}{Data Frame. A three column data frame. First column is the operation outcome.
Second column are the predicted probabilities from the risk model. Third column can be either the
 predicted probabilities from the risk model or average outcome.}

\item{RA}{Double. Odds ratio of death under the alternative hypotheses. Detecting deterioration
in performance with increased mortality risk by doubling the odds Ratio \code{RA = 2}. Detecting
 improvement in performance with decreased mortality risk by halving the odds ratio of death
 \code{RA = 1/2}. Odds ratio of death under the null hypotheses is \code{1}.}

\item{yemp}{Logical. If \code{TRUE}, use emirical outcome values, else use model.}
}
\value{
Returns a single value which is the approximate optimal \code{k}.
}
\description{
Compute approximate optimal k.
}
\details{
Formula deterioration:  \deqn{ k{det} = \frac{R{A} - 1 - log(R{A})}{log(R{A})}\bar{p} ,
R{A} > 1    }
         Formula improvement:    \deqn{ k{imp} = \frac{1 - R{A} + log(R{A})}{log(R{A})}\bar{p} ,
R{A} < 1    }
}
\examples{
\dontrun{
library(vlad)
library(dplyr)
data("cardiacsurgery", package = "spcadjust")

## preprocess data to 30 day mortality
SALL <- cardiacsurgery \%>\% rename(s = Parsonnet) \%>\%
  mutate(y = ifelse(status == 1 & time <= 30, 1, 0),
         phase = factor(ifelse(date < 2*365, "I", "II")))
SI <- subset(SALL, phase == "I")
GLM <- glm(y ~ s, data = SI, family = "binomial")
pi1 <- predict(GLM, type = "response", newdata = data.frame(s = SI$s))
pmix <- data.frame(SI$y, pi1, pi1)

## (Deterioration)
optimal_k(pmix = pmix, RA = 2)

## manually find optimal k for detecting deterioration
RA <- 2
pbar <- mean(pmix$pi1)
kopt <- pbar * ( RA - 1 - log(RA) ) / log(RA)

all.equal(kopt, optimal_k(pmix = pmix, RA = 2))

## (Improvement)
optimal_k(pmix = pmix, RA = 1/2)

## manually find optimal k for detecting improvement
RA <- 1/2
pbar <- mean(pmix$pi1)
kopt <- pbar * ( 1 - RA + log(RA) ) / log(RA)

all.equal(kopt, optimal_k(pmix = pmix, RA = 1/2))
}
}
\references{
Wittenberg P, Gan FF, Knoth S (2018).
A simple signaling rule for variable life-adjusted display derived from
an equivalent risk-adjusted CUSUM chart.
\emph{Statistics in Medicine}, \strong{37}(16), pp 2455--2473.
}
\author{
Philipp Wittenberg
}
