% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cv_vim.R
\name{cv_vim}
\alias{cv_vim}
\title{Nonparametric Variable Importance Estimates using Cross-validation}
\usage{
cv_vim(
  Y,
  X,
  f1,
  f2,
  indx = 1,
  V = length(unique(folds)),
  folds = NULL,
  stratified = FALSE,
  weights = rep(1, length(Y)),
  type = "r_squared",
  run_regression = TRUE,
  SL.library = c("SL.glmnet", "SL.xgboost", "SL.mean"),
  alpha = 0.05,
  delta = 0,
  scale = "identity",
  na.rm = FALSE,
  ...
)
}
\arguments{
\item{Y}{the outcome.}

\item{X}{the covariates.}

\item{f1}{the predicted values on validation data from a flexible estimation technique regressing Y on X in the training data; a list of length V, where each object is a set of predictions on the validation data.}

\item{f2}{the predicted values on validation data from a flexible estimation technique regressing the fitted values in \code{f1} on X withholding the columns in \code{indx}; a list of length V, where each object is a set of predictions on the validation data.}

\item{indx}{the indices of the covariate(s) to calculate variable importance for; defaults to 1.}

\item{V}{the number of folds for cross-validation, defaults to 10.}

\item{folds}{the folds to use, if f1 and f2 are supplied. A list of length two; the first element provides the outer folds (for hypothesis testing), while the second element is a list providing the inner folds (for cross-validation).}

\item{stratified}{if run_regression = TRUE, then should the generated folds be stratified based on the outcome (helps to ensure class balance across cross-validation folds)}

\item{weights}{weights for the computed influence curve (e.g., inverse probability weights for coarsened-at-random settings)}

\item{type}{the type of parameter (e.g., ANOVA-based is \code{"anova"}).}

\item{run_regression}{if outcome Y and covariates X are passed to \code{cv_vim}, and \code{run_regression} is \code{TRUE}, then Super Learner will be used; otherwise, variable importance will be computed using the inputted fitted values.}

\item{SL.library}{a character vector of learners to pass to \code{SuperLearner}, if \code{f1} and \code{f2} are Y and X, respectively. Defaults to \code{SL.glmnet}, \code{SL.xgboost}, and \code{SL.mean}.}

\item{alpha}{the level to compute the confidence interval at. Defaults to 0.05, corresponding to a 95\% confidence interval.}

\item{delta}{the value of the \eqn{\delta}-null (i.e., testing if importance < \eqn{\delta}); defaults to 0.}

\item{scale}{should CIs be computed on original ("identity") or logit ("logit") scale?}

\item{na.rm}{should we remove NA's in the outcome and fitted values in computation? (defaults to \code{FALSE})}

\item{...}{other arguments to the estimation tool, see "See also".}
}
\value{
An object of class \code{vim}. See Details for more information.
}
\description{
Compute estimates and confidence intervals for the
nonparametric variable importance parameter of interest, using cross-validation.
This essentially involves splitting the data into V train/test splits; train the learners on the training data, evaluate importance on the test data; and average over these splits.
}
\details{
See the paper by Williamson, Gilbert, Simon, and Carone for more
details on the mathematics behind this function, and the validity
of the confidence intervals.
In the interest of transparency, we return most of the calculations
within the \code{vim} object. This results in a list containing:
\itemize{
 \item{call}{ - the call to \code{cv_vim}}
 \item{s}{ - the column(s) to calculate variable importance for}
 \item{SL.library}{ - the library of learners passed to \code{SuperLearner}}
 \item{full_fit}{ - the fitted values of the chosen method fit to the full data (a list, for train and test data)}
 \item{red_fit}{ - the fitted values of the chosen method fit to the reduced data (a list, for train and test data)}
 \item{est}{ - the estimated variable importance}
 \item{naive}{ - the naive estimator of variable importance}
 \item{naives}{ - the naive estimator on each fold}
 \item{updates}{ - the influence curve-based update for each fold}
 \item{se}{ - the standard error for the estimated variable importance}
 \item{ci}{ - the \eqn{(1-\alpha) \times 100}\% confidence interval for the variable importance estimate}
 \item{full_mod}{ - the object returned by the estimation procedure for the full data regression (if applicable)}
 \item{red_mod}{ - the object returned by the estimation procedure for the reduced data regression (if applicable)}
 \item{alpha}{ - the level, for confidence interval calculation}
 \item{folds}{ - the folds used for hypothesis testing and cross-validation}
 \item{y}{ - the outcome}
 \item{weights}{ - the weights}
 \item{mat}{- a tibble with the estimate, SE, CI, hypothesis testing decision, and p-value}
}
}
\examples{
\donttest{
## don't test because this can take a long time to run
library(SuperLearner)
library(ranger)
n <- 100
p <- 2
## generate the data
x <- data.frame(replicate(p, stats::runif(n, -5, 5)))

## apply the function to the x's
smooth <- (x[,1]/5)^2*(x[,1]+7)/5 + (x[,2]/3)^2

## generate Y ~ Normal (smooth, 1)
y <- as.matrix(smooth + stats::rnorm(n, 0, 1))

## set up a library for SuperLearner
learners <- c("SL.mean", "SL.ranger")

## -----------------------------------------
## using Super Learner (with a small number of folds, for illustration only)
## -----------------------------------------
set.seed(4747)
est <- cv_vim(Y = y, X = x, indx = 2, V = 2,
type = "r_squared", run_regression = TRUE,
SL.library = learners, cvControl = list(V = 5), alpha = 0.05)

## ------------------------------------------
## doing things by hand, and plugging them in (with a small number of folds, for illustration only)
## ------------------------------------------
## set up the folds
indx <- 2
V <- 2
set.seed(4747)
folds <- rep(seq_len(V), length = n)
folds <- sample(folds)
## get the fitted values by fitting the super learner on each pair
fhat_ful <- list()
fhat_red <- list()
for (v in 1:V) {
    ## fit super learner
    fit <- SuperLearner::SuperLearner(Y = y[folds != v, , drop = FALSE],
     X = x[folds != v, , drop = FALSE], SL.library = learners, cvControl = list(V = 5))
    fitted_v <- SuperLearner::predict.SuperLearner(fit)$pred
    ## get predictions on the validation fold
    fhat_ful[[v]] <- SuperLearner::predict.SuperLearner(fit,
     newdata = x[folds == v, , drop = FALSE])$pred
    ## fit the super learner on the reduced covariates
    red <- SuperLearner::SuperLearner(Y = fitted_v,
     X = x[folds != v, -indx, drop = FALSE], SL.library = learners, cvControl = list(V = 5))
    ## get predictions on the validation fold
    fhat_red[[v]] <- SuperLearner::predict.SuperLearner(red,
     newdata = x[folds == v, -indx, drop = FALSE])$pred
}
est <- cv_vim(Y = y, f1 = fhat_ful, f2 = fhat_red, indx = 2,
V = V, folds = folds, type = "r_squared", run_regression = FALSE, alpha = 0.05)
}

}
\seealso{
\code{\link[SuperLearner]{SuperLearner}} for specific usage of the \code{SuperLearner} function and package.
}
