% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/build_run_modify.R
\name{umxIP}
\alias{umxIP}
\title{umxIP: Build and run an Independent Pathway twin model}
\usage{
umxIP(
  name = "IP",
  selDVs,
  dzData,
  mzData,
  sep = NULL,
  nFac = c(a = 1, c = 1, e = 1),
  data = NULL,
  zyg = "zygosity",
  type = c("Auto", "FIML", "cov", "cor", "WLS", "DWLS", "ULS"),
  allContinuousMethod = c("cumulants", "marginals"),
  dzAr = 0.5,
  dzCr = 1,
  correlatedA = FALSE,
  numObsDZ = NULL,
  numObsMZ = NULL,
  autoRun = getOption("umx_auto_run"),
  tryHard = c("no", "yes", "ordinal", "search"),
  optimizer = NULL,
  equateMeans = TRUE,
  weightVar = NULL,
  addStd = TRUE,
  addCI = TRUE,
  freeLowerA = FALSE,
  freeLowerC = FALSE,
  freeLowerE = FALSE
)
}
\arguments{
\item{name}{The name of the model (defaults to "IP").}

\item{selDVs}{The base names of the variables to model. note: Omit suffixes - just "dep" not c("dep_T1", "dep_T2")}

\item{dzData}{The DZ dataframe.}

\item{mzData}{The MZ dataframe.}

\item{sep}{The suffix for twin 1 and twin 2. e.g. selDVs= "dep", sep= "_T" -> c("dep_T1", "dep_T2")}

\item{nFac}{How many common factors for a, c, and e. If one number is given, applies to all three.}

\item{data}{If provided, dzData and mzData are treated as levels of zyg to select() MZ and DZ data sets (default = NULL)}

\item{zyg}{If data provided, this column is used to select rows by zygosity (Default = "zygosity")}

\item{type}{Analysis method one of c("Auto", "FIML", "cov", "cor", "WLS", "DWLS", "ULS")}

\item{allContinuousMethod}{"cumulants" or "marginals". Used in all-continuous WLS data to determine if a means model needed.}

\item{dzAr}{The DZ genetic correlation (defaults to .5, vary to examine assortative mating).}

\item{dzCr}{The DZ "C" correlation (defaults to 1: set to .25 to make an ADE model).}

\item{correlatedA}{Whether factors are allowed to correlate (not implemented yet: FALSE).}

\item{numObsDZ}{= For cov data, the number of DZ pairs.}

\item{numObsMZ}{= For cov data, the number of MZ pairs.}

\item{autoRun}{Whether to run and return the model (default), or just to create and return without running.}

\item{tryHard}{Whether to tryHard (default 'no' uses normal mxRun). options: "mxTryHard", "mxTryHardOrdinal", or "mxTryHardWideSearch"}

\item{optimizer}{optionally set the optimizer (default NULL does nothing).}

\item{equateMeans}{Whether to equate the means across twins (defaults to TRUE).}

\item{weightVar}{If a weighting variable is provided, a vector objective will be used to weight the data. (default = NULL).}

\item{addStd}{Whether to add algebras for a standardized model (defaults to TRUE).}

\item{addCI}{Whether to add CIs (defaults to TRUE).}

\item{freeLowerA}{ignore: Whether to leave the lower triangle of A free (default = FALSE).}

\item{freeLowerC}{ignore: Whether to leave the lower triangle of C free (default = FALSE).}

\item{freeLowerE}{ignore: Whether to leave the lower triangle of E free (default = FALSE).}
}
\value{
\itemize{
\item \code{\link[=mxModel]{mxModel()}}
}
}
\description{
Make a 2-group Independent Pathway twin model.

The independent-pathway model  (aka "biometric model" (McArdle and Goldsmith, 1990) proposes that \code{A},
\code{C}, and \code{E} components act directly on the manifest or measured phenotypes. This contrasts with
the \code{\link[=umxCP]{umxCP()}} model, in which these influences are collected on a hypothesized or latent causal
variable, which is manifested in the measured phenotypes.

The following figure shows the IP model diagrammatically:

\if{html}{\figure{IP.svg}{options: width=50\% alt="Figure: IP model"}}
\if{latex}{\figure{IP.pdf}{options: width=7cm}}

As can be seen, each phenotype also by default has A, C, and E influences specific to that phenotype.

Features of the model include the ability to include add more one set of independent pathways, different numbers
of pathways for a, c, and e, as well the ability to use ordinal data, and different fit functions, e.g. WLS.

\strong{note}: The function \code{\link[=umx_set_optimization_options]{umx_set_optimization_options()}} allows users to see and set \code{mvnRelEps} and \code{mvnMaxPointsA}
mvnRelEps defaults to .005. For ordinal models, you might find that '0.01' works better.
}
\details{
Like the \code{\link[=umxACE]{umxACE()}} model, the IP model decomposes phenotypic variance
into additive genetic (A), unique environmental (E) and, optionally, either
common or shared-environment (C) or
non-additive genetic effects (D).

Unlike the Cholesky, these factors do not act directly on the phenotype. Instead latent A,
C, and E influences impact on one or more latent common factors which, in turn, account for
variance in the phenotypes (see Figure).

\strong{Data Input}
Currently, \code{umxIP} accepts only raw data. This may change in future versions. You can
choose other fit functions, e.g. WLS.

\strong{Ordinal Data}

In an important capability, the model transparently handles ordinal (binary or multi-level
ordered factor data) inputs, and can handle mixtures of continuous, binary, and ordinal
data in any combination.

\strong{Additional features}

\code{umxIP} supports varying the DZ genetic association (defaulting to .5)
to allow exploring assortative mating effects, as well as varying the DZ \dQuote{C} factor
from 1 (the default for modeling family-level effects shared 100\% by twins in a pair),
to .25 to model dominance effects.

\strong{Matrices and Labels in IP model}

A good way to see which matrices are used in umxIP is to run an example model and plot it.

All the shared matrices are in the model "top".

Matrices \code{as}, \code{cs}, and \code{es} contain the path loadings specific to each variable on their diagonals.

To see the 'as' values, you can simply execute:

m1$top#as$values

m1$top#as$labels

m1$top#as$free

Labels relevant to modifying the specific loadings take the form "as_r1c1", "as_r2c2" etc.

The independent-pathway loadings on the manifests are in matrices \code{a_ip}, \code{c_ip}, \code{e_ip}.

Less commonly-modified matrices are the mean matrix \code{expMean}.
This has 1 row, and the columns are laid out for each variable
for twin 1, followed by each variable for twin 2.

So, in a model where the means for twin 1 and twin 2 had been equated
(set = to T1), you could make them independent again with this line:

\code{m1$top$expMean$labels[1,4:6] = c("expMean_r1c4", "expMean_r1c5", "expMean_r1c6")}
}
\examples{
\dontrun{
require(umx)
data(GFF)
mzData = subset(GFF, zyg_2grp == "MZ")
dzData = subset(GFF, zyg_2grp == "DZ")
selDVs = c("gff","fc","qol","hap","sat","AD") # These will be expanded into "gff_T1" "gff_T2" etc.
m1 =    umxIP(selDVs = selDVs, sep = "_T", dzData = dzData, mzData = mzData)

# WLS example: Use "marginals" method to enable all continuous data with missingness.
m3 = umxIP(selDVs = selDVs, sep = "_T", dzData = dzData, mzData = mzData, 
	type = "DWLS", allContinuousMethod='marginals')
# omit missing to enable default WLS method to work on all continuous data
dzD = na.omit(dzData[, tvars(selDVs, "_T")])
mzD = na.omit(dzData[, tvars(selDVs, "_T")])
m4 = umxIP(selDVs = selDVs, sep = "_T", dzData = dzD, mzData = mzD, type = "DWLS")

# ====================================================================
# = Try with a non-default number of a, c, and e independent factors =
# ====================================================================
nFac = c(a = 2, c = 1, e = 1)
m2 = umxIP(selDVs = selDVs, sep = "_T", dzData = dzData, mzData = mzData, nFac = nFac, 
	tryHard = "yes")
umxCompare(m1, m2)
}
}
\references{
\itemize{
\item Kendler, K. S., Heath, A. C., Martin, N. G., & Eaves, L. J. (1987). Symptoms of anxiety and symptoms of depression.
Same genes, different environments? \emph{Archives of General Psychiatry}, \strong{44}, 451-457. \doi{10.1001/archpsyc.1987.01800170073010}.
\item McArdle, J. J., & Goldsmith, H. H. (1990). Alternative common factor models for multivariate biometric analyses.
\emph{Behavior Genetics}, \strong{20}, 569-608. \doi{10.1007/BF01065873}.
\item \url{https://github.com/tbates/umx}
}
}
\seealso{
\itemize{
\item \code{\link[=plot]{plot()}}, \code{\link[=umxSummary]{umxSummary()}}, \code{\link[=umxCP]{umxCP()}}
}

Other Twin Modeling Functions: 
\code{\link{power.ACE.test}()},
\code{\link{umxACEcov}()},
\code{\link{umxACEv}()},
\code{\link{umxACE}()},
\code{\link{umxCP}()},
\code{\link{umxDiffMZ}()},
\code{\link{umxDiscTwin}()},
\code{\link{umxDoCp}()},
\code{\link{umxDoC}()},
\code{\link{umxGxE_window}()},
\code{\link{umxGxEbiv}()},
\code{\link{umxGxE}()},
\code{\link{umxReduceACE}()},
\code{\link{umxReduceGxE}()},
\code{\link{umxReduce}()},
\code{\link{umxRotate.MxModelCP}()},
\code{\link{umxSexLim}()},
\code{\link{umxSimplex}()},
\code{\link{umxSummarizeTwinData}()},
\code{\link{umxSummaryACEv}()},
\code{\link{umxSummaryACE}()},
\code{\link{umxSummaryDoC}()},
\code{\link{umxSummaryGxEbiv}()},
\code{\link{umxSummarySexLim}()},
\code{\link{umxSummarySimplex}()},
\code{\link{umxTwinMaker}()},
\code{\link{umx}}
}
\concept{Twin Modeling Functions}
