% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/umx_build_umxACEv.R
\name{umxACEv}
\alias{umxACEv}
\title{Build and run 2-group uni- or multi-variate ACE models based on VARIANCE (not paths).}
\usage{
umxACEv(name = "ACEv", selDVs, selCovs = NULL, sep = NULL,
  type = c("Auto", "FIML", "cov", "cor", "WLS", "DWLS", "ULS"), dzData,
  mzData, dzAr = 0.5, dzCr = 1, addStd = TRUE, addCI = TRUE,
  numObsDZ = NULL, numObsMZ = NULL, boundDiag = NULL,
  weightVar = NULL, equateMeans = TRUE, bVector = FALSE,
  thresholds = c("deviationBased"), covMethod = c("fixed", "random"),
  autoRun = getOption("umx_auto_run"), tryHard = c("no", "mxTryHard",
  "mxTryHardOrdinal", "mxTryHardWideSearch"), optimizer = NULL)
}
\arguments{
\item{name}{The name of the model (defaults to"ACE").}

\item{selDVs}{The variables to include from the data: preferably, just "dep" not c("dep_T1", "dep_T2").}

\item{selCovs}{(optional) covariates to include from the data (do not include sep in names)}

\item{sep}{The separator in twin var names, often "_T" in vars like "dep_T1". Simplifies selDVs.}

\item{type}{Analysis method one of c("Auto", "FIML", "cov", "cor", "WLS", "DWLS", "ULS").}

\item{dzData}{The DZ dataframe.}

\item{mzData}{The MZ dataframe.}

\item{dzAr}{The DZ genetic correlation (defaults to .5, vary to examine assortative mating).}

\item{dzCr}{The DZ "C" correlation (defaults to 1: set to .25 to make an ADE model).}

\item{addStd}{Whether to add the algebras to compute a std model (defaults to TRUE).}

\item{addCI}{Whether to add intervals to compute CIs (defaults to TRUE).}

\item{numObsDZ}{= Number of DZ twins: Set this if you input covariance data.}

\item{numObsMZ}{= Number of MZ twins: Set this if you input covariance data.}

\item{boundDiag}{= Numeric lbound for diagonal of the a, c, and e matrices. Default = NULL (no bound)}

\item{weightVar}{= If provided, a vector objective will be used to weight the data. (default = NULL).}

\item{equateMeans}{Whether to equate the means across twins (defaults to TRUE).}

\item{bVector}{Whether to compute row-wise likelihoods (defaults to FALSE).}

\item{thresholds}{How to implement ordinal thresholds c("deviationBased").}

\item{covMethod}{How to treat covariates: "fixed" (default) or "random".}

\item{autoRun}{Whether to run the model, and return that (default), or just to create it and return without running.}

\item{tryHard}{optionally tryHard (default 'no' uses normal mxRun). c("no", "mxTryHard", "mxTryHardOrdinal", "mxTryHardWideSearch")}

\item{optimizer}{Optionally set the optimizer (default NULL does nothing).}
}
\value{
- \code{\link{mxModel}} of subclass mxModel.ACE
}
\description{
A common task in twin modeling involves using the genetic and environmental differences 
between large numbers of pairs of mono-zygotic (MZ) and di-zygotic (DZ) twins reared together
to model the genetic and environmental structure of one, or, typically, several phenotypes
(measured behaviors).
}
\details{
The ACE variance-based model decomposes phenotypic variance into additive genetic (A),
unique environmental (E) and, optionally, either common environment (shared-environment, C) or 
non-additive genetic effects (D). Scroll down to details for how to use the function, a figure
and multiple examples.

The following figure shows the A components of a trivariate ACEv model:

\if{html}{\figure{ACEv.png}{options: width="50\%" alt="Figure: ACEv.png"}}
\if{latex}{\figure{ACEv.pdf}{options: width=7cm}}

\emph{NOTE}: This function does not use the Cholesky decomposition. Instead it directly models variance.
This ensures unbiased type-I error rates. It means that occasionally
estimates of variance may be negative. This should be used as an occasion to inspect you model
choices and data. `umxACEv` can be used as a base model to validate the ACE Cholesky model, 
a core model in behavior genetics (Neale and Cardon, 1992).


\strong{Data Input}
The function flexibly accepts raw data, and also summary covariance data 
(in which case the user must also supple numbers of observations for the two input data sets).

\strong{Ordinal Data}
In an important capability, the model transparently handles ordinal (binary or multi-level
ordered factor data) inputs, and can handle mixtures of continuous, binary, and ordinal
data in any combination. 

The function also supports weighting of individual data rows. In this case,
the model is estimated for each row individually, then each row likelihood
is multiplied by its weight, and these weighted likelihoods summed to form
the model-likelihood, which is to be minimized.
This feature is used in the non-linear GxE model functions.

\strong{Additional features}
The umxACEv function supports varying the DZ genetic association (defaulting to .5)
to allow exploring assortative mating effects, as well as varying the DZ \dQuote{C} factor
from 1 (the default for modeling family-level effects shared 100% by twins in a pair),
to .25 to model dominance effects.

\emph{note}: Only one of C or D may be estimated simultaneously. This restriction reflects the lack
of degrees of freedom to simultaneously model C and D with only MZ and DZ twin pairs (Eaves et al. 1978 p267).
}
\examples{

# ==============================
# = Univariate model of weight =
# ==============================
require(umx)
data(twinData) # ?twinData from Australian twins.

# Things to note: ACE model of weight will return a NEGATIVE variance in C.
#  This is exactly why we have ACEv! It suggests we need a different model
#  In this case: ADE.
# Other things to note:
# 1. umxACEv can figure out variable names: provide "sep", and selVars. 
#    Function generates: "wt" -> "wt1" "wt2"
# 2. umxACEv picks the variables it needs from the data.

selDVs = "wt"
mzData <- twinData[twinData$zygosity \%in\% "MZFF", ]
dzData <- twinData[twinData$zygosity \%in\% "DZFF", ]
m1 = umxACEv(selDVs = selDVs, sep = "", dzData = dzData, mzData = mzData)

# ========================================================
# = Evidence for dominance ? (DZ correlation set to .25) =
# ========================================================
m2 = umxACEv("ADE", selDVs = selDVs, sep = "", dzData = dzData, mzData = mzData, dzCr = .25)
# note: the underlying matrices are still called A, C, and E.
# I catch this in the summary table, so columns are labeled A, D, and E.
# However, currently, the plot will say A, C, E.

# We can modify this model, dropping dominance component (still called C), 
# and see a comparison:
m3 = umxModify(m2, update = "C_r1c1", comparison = TRUE, name="AE")
# =========================================================
# = Well done! Now you can make modify twin models in umx =
# =========================================================

# ============================
# = How heritable is height? =
# ============================
# 
# Note: Height has a small variance. umx can typically picks good starts,
#    but scaling is advisable.
# 
require(umx)
data(twinData) # ?twinData from Australian twins.
# Pick the variables
selDVs = "ht"
twinData$ht1 = twinData$ht1*10
twinData$ht2 = twinData$ht2*10
mzData <- twinData[twinData$zygosity \%in\% "MZFF", ]
dzData <- twinData[twinData$zygosity \%in\% "DZFF", ]
m1 = umxACEv(selDVs = selDVs, sep = "", dzData = dzData, mzData = mzData)
umxSummary(m1, std = FALSE) # unstandardized
# tip: with report = "html", umxSummary can print the table to your browser!
plot(m1)

# ========================================================
# = Evidence for dominance ? (DZ correlation set to .25) =
# ========================================================
m2 = umxACEv("ADE", selDVs = selDVs, sep="", dzData = dzData, mzData = mzData, dzCr = .25)
umxCompare(m2, m1) # Is ADE better?
umxSummary(m2, comparison = m1) # nb: though this is ADE, matrices are still called A,C,E

# We can modify this model, dropping shared environment, and see a comparison:
m3 = umxModify(m2, update = "C_r1c1", comparison = TRUE, name = "AE")

# =====================================
# = Bivariate height and weight model =
# =====================================

data(twinData)
twinData$ht1 = twinData$ht1 * 1000 # Convert m to mm
twinData$ht2 = twinData$ht2 * 1000
mzData = twinData[twinData$zygosity \%in\% c("MZFF", "MZMM"), ]
dzData = twinData[twinData$zygosity \%in\% c("DZFF", "DZMM", "DZOS"), ]
mzData = mzData[1:80, ] # Quicker run to keep CRAN happy
dzData = dzData[1:80, ]
m1 = umxACEv(selDVs = c("ht", "wt"), sep = '', dzData = dzData, mzData = mzData)

# ===================
# = Ordinal example =
# ===================
require(umx)
data(twinData)
# Cut bmi column to form ordinal obesity variables
ordDVs = c("obese1", "obese2")
selDVs = c("obese")
obesityLevels = c('normal', 'overweight', 'obese')
cutPoints = quantile(twinData[, "bmi1"], probs = c(.5, .2), na.rm = TRUE)
twinData$obese1 = cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 = cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
# Make the ordinal variables into mxFactors (ensure ordered is TRUE, and require levels)
twinData[, ordDVs] = mxFactor(twinData[, ordDVs], levels = obesityLevels)
mzData = twinData[twinData$zygosity \%in\% "MZFF", ][1:80,] # 80 pairs for speed
dzData = twinData[twinData$zygosity \%in\% "DZFF", ][1:80,]
str(mzData) # make sure mz, dz, and t1 and t2 have the same levels!
m1 = umxACEv(selDVs = selDVs, dzData = dzData, mzData = mzData, sep = '')
umxSummary(m1)

# ============================================
# = Bivariate continuous and ordinal example =
# ============================================
data(twinData)
selDVs = c("wt", "obese")
# Cut bmi column to form ordinal obesity variables
ordDVs = c("obese1", "obese2")
obesityLevels = c('normal', 'overweight', 'obese')
cutPoints = quantile(twinData[, "bmi1"], probs = c(.5, .2), na.rm = TRUE)
twinData$obese1 = cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 = cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 

# Make the ordinal variables into mxFactors (ensure ordered is TRUE, and require levels)
twinData[, ordDVs] = mxFactor(twinData[, ordDVs], levels = obesityLevels)

# umxACEv can trim out unused variables on its own
mzData = twinData[twinData$zygosity \%in\% "MZFF", ]
dzData = twinData[twinData$zygosity \%in\% "DZFF", ]
m1 = umxACEv(selDVs = selDVs, dzData = dzData, mzData = mzData, sep = '')
plot(m1)

# =======================================
# = Mixed continuous and binary example =
# =======================================
require(umx)
data(twinData)
# Cut to form category of 20\% obese subjects
# and make into mxFactors (ensure ordered is TRUE, and require levels)
cutPoints = quantile(twinData[, "bmi1"], probs = .2, na.rm = TRUE)
obesityLevels = c('normal', 'obese')
twinData$obese1 = cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 = cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
ordDVs = c("obese1", "obese2")
twinData[, ordDVs] = mxFactor(twinData[, ordDVs], levels = obesityLevels)

selDVs = c("wt", "obese")
mzData = twinData[twinData$zygosity \%in\% "MZFF", ]
dzData = twinData[twinData$zygosity \%in\% "DZFF", ]
\dontrun{
m1 = umxACEv(selDVs = selDVs, dzData = dzData, mzData = mzData, sep = '')
umxSummary(m1)
}

# ===================================
# Example with covariance data only =
# ===================================

require(umx)
data(twinData)
selDVs = c("wt")
mz = cov(twinData[twinData$zygosity \%in\% "MZFF", tvars(selDVs, "")], use = "complete")
dz = cov(twinData[twinData$zygosity \%in\% "DZFF", tvars(selDVs, "")], use = "complete")
m1 = umxACEv(selDVs = selDVs, sep= "", dzData = dz, mzData= mz, numObsDZ= 569, numObsMZ= 351)
umxSummary(m1, std = FALSE)

}
\references{
- Eaves, L. J., Last, K. A., Young, P. A., & Martin, N. G. (1978). Model-fitting approaches 
to the analysis of human behaviour. Heredity, 41(3), 249-320. \url{https://www.nature.com/articles/hdy1978101.pdf}
}
\seealso{
Other Twin Modeling Functions: \code{\link{umxACE_cov_fixed}},
  \code{\link{umxACEcov}}, \code{\link{umxACEnew}},
  \code{\link{umxACE}}, \code{\link{umxCPold}},
  \code{\link{umxCP}}, \code{\link{umxGxE_window}},
  \code{\link{umxGxEbiv}}, \code{\link{umxGxE}},
  \code{\link{umxIPnew}}, \code{\link{umxIP}},
  \code{\link{umxSexLim}}, \code{\link{umxSimplex}},
  \code{\link{umxSummaryACEcov}},
  \code{\link{umxSummaryACEv}},
  \code{\link{umxSummaryACE}}, \code{\link{umxSummaryCP}},
  \code{\link{umxSummaryGxEbiv}},
  \code{\link{umxSummaryGxE}}, \code{\link{umxSummaryIP}},
  \code{\link{umxSummarySexLim}},
  \code{\link{umxSummarySimplex}}, \code{\link{umx}},
  \code{\link{xmu_twin_check}}
}
\concept{Twin Modeling Functions}
