% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_resource.R
\name{get_resource}
\alias{get_resource}
\title{Get Data From Trello API}
\usage{
get_resource(
  parent = NULL,
  child = NULL,
  id = NULL,
  token = NULL,
  query = NULL,
  url = NULL,
  filter = NULL,
  limit = 100,
  on.error = c("stop", "warn", "message"),
  retry.times = 1,
  handle = NULL,
  verbose = FALSE,
  response,
  paging,
  bind.rows
)
}
\arguments{
\item{parent}{Parent resource, e.g. \code{"board"} or \code{NULL}.}

\item{child}{Child resource, eg. \code{"card"} or \code{NULL}.}

\item{id}{Resource ID or \code{NULL}.}

\item{token}{An object of class \code{"Trello_API_token"}, a path or \code{NULL}.
\itemize{
\item If a \code{Token}, it is passed as is.
\item If \code{NULL} and a cache file called \code{".httr-oauth"} exists, the newest token
is read from it. If the file is not found, an error is thrown.
\item If a character vector of length 1, it will be used as an alternative path
to the cache file.
}}

\item{query}{Named list of key-value pairs, see \code{\link[httr:GET]{httr::GET()}} for details.}

\item{url}{Url for the GET request. Can be \code{NULL} if \code{parent} is specified,
or a combination of \code{parent}, \code{child} and \code{id} is provided.}

\item{filter}{Defaults to \code{"all"} which includes both open and archived cards
or all action types, depending on what resource is requested.}

\item{limit}{Defaults to \code{100}. Set to \code{Inf} to get everything.}

\item{on.error}{Whether to \code{"stop"}, \code{"warn"} or \code{"message"} on API error.}

\item{retry.times}{How many times to re-try when a request fails. Defaults
to 1.}

\item{handle}{The handle to use with this request, see \code{\link[httr:RETRY]{httr::RETRY()}}.}

\item{verbose}{Set to \code{TRUE} for verbose output.}

\item{response, paging, bind.rows}{Deprecated.}
}
\value{
A data frame with API responses.
}
\description{
Fetch resources using Trello API.
}
\section{Request limits}{


At maximum, the API can retrieve 1000 results in a single call. Setting
\code{limit > 1000} will activate paging. When paging is used, the request will
be issued repeatedly, retrieving new batch of results each time until
the \code{limit} is reached or there is nothing else to fetch. Results are fetched
chronologically, ie. newest results are retrieved first (eg. newest cards).
Use \code{limit = Inf} to make sure you get all.
}

\section{Errors}{


If the request fails, server error messages are reprinted on the console.
Depending on the value of \code{on.error}, the request call can throw an error
in R (this is the default), or can issue a warning/message. If the latter,
the function returns a data frame containing the failed URL, HTTP status
and an informative message (produced by the server).
}

\section{Results}{


The API returns JSON objects which are parsed using \code{\link[jsonlite:fromJSON]{jsonlite::fromJSON()}}.
Non-JSON results throw an error, but these should never happen anyway. The
result is always a data frame, or a \code{tibble} if the package is installed.
}

\section{Filter}{


Both \code{filter} and \code{limit} exist as explicitly defined arguments, but you can
ignore them in favor of supplying their values as query parameters, eg.
\code{query = list(filter = "filter_value", limit = "limit_value")}.
}

\examples{

# Public boards can be accessed without authorization, so there is no need
# to create a token, just the board id:
url = "https://trello.com/b/wVWPK9I4/r-client-for-the-trello-api"
bid = get_id_board(url)

# Getting resources from the whole board. `filter="all"` fetches archived
# cards as well.
labels = get_board_labels(bid)
cards = get_board_cards(bid, filter = "all")

# It is possible to call `get_resource()` directly:
lists = get_resource(parent = "board", child = "lists", id = bid)

# As with boards, cards can be queried for particular resources, in this case
# to obtain custom fields:
card = cards$id[5]
acts = get_card_fields(card)

# Set `limit` to specify the number of results. Pagination will be used
# whenever limit exceeds 1000. Use `limit=Inf` to make sure you get all.

\dontrun{
all_actions = get_board_actions(bid, limit = Inf)
}

# For private and team boards, a secure token is required:

\dontrun{
key = Sys.getenv("MY_TRELLO_KEY")
secret = Sys.getenv("MY_TRELLO_SECRET")

token = get_token("my_app", key = key, secret = secret,
                  scope = c("read", "write"))

# Token is now cached, no need to pass it explicitly.
cards_open = get_board_cards(board_id, filter = "open")
}
}
\seealso{
\code{\link[=get_token]{get_token()}}, \code{\link[=get_id]{get_id()}}, \code{\link[httr:GET]{httr::GET()}}, \code{\link[jsonlite:fromJSON]{jsonlite::fromJSON()}}
}
