% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/speed.R
\name{TrajVelocity}
\alias{TrajVelocity}
\title{Velocity of a trajectory}
\usage{
TrajVelocity(trj, diff = c("central", "forward", "backward"))
}
\arguments{
\item{trj}{Trajectory whose velocity is to be calculated.}

\item{diff}{Type of difference to be calculated, one of "central" (the
default), "forward" or "backward".}
}
\value{
A vector of complex numbers representing the velocity at each point
  along the trajectory. The modulus (\code{Mod(v)}) is the magnitude of the
  velocity, i.e. the speed; the argument (\code{Arg(v)}) is the direction of
  the velocity; the real part (\code{Re(v)}) is velocity in the X direction;
  and the imaginary part (\code{Im(v)}) is velocity in the Y direction. The
  vector has an attribute, \code{trj}, with the trajectory as its value. If
  \code{diff} is \code{"central"}, the first and last velocity values are
  \code{NA} since velocity cannot be calculated for them. If \code{diff} is
  \code{"forward"}, the last value will be NA, and if \code{diff} is
  \code{"backward"}, the first value will be NA.
}
\description{
The velocity is approximated at each point of the trajectory using
first-order finite differences. Central, forward or backward differences can
be used. Central differences yield a more accurate approximation if the
velocity is smooth. As a practical guide, if velocity doesn't change much
between steps, use central differences. If it changes substantially (and not
just as an artifact of recording noise), then use either forward or backward
differences.
}
\details{
Intuitively, think of the central difference velocity at a point as the mean
of the velocities of the two adjacent steps. Forward difference velocity is
the velocity of the step starting at the point. Backward difference is the
velocity of the step ending at the point.
}
\examples{
set.seed(11)
trj <- TrajGenerate(100)
# calculate velocity
vel <- TrajVelocity(trj)

# Obtain speed over time, with NAs removed
speed <- na.omit(data.frame(speed = Mod(vel), time = trj$time))

plot(speed ~ time, speed, type = 'l')

}
\seealso{
\code{\link{TrajAcceleration}} for calculating acceleration;
\code{\link{TrajResampleTime}} and \code{\link{TrajRediscretize}} to resample
a trajectory to fixed time or length steps; \code{\link{TrajSpeedIntervals}}
for calculating when speed crosses some threshold; Finite differences on
\href{https://en.wikipedia.org/wiki/Finite_difference}{Wikipedia}.
}
