#' Bounding box generator
#'
#' Swiss army knife for bounding boxes. Modify an existing bounding box or create a new bounding box from scratch. See details.
#'
#' An existing bounding box (defined by \code{x}) can be modified as follows:
#' \itemize{
#' \item Using the extension factor \code{ext}.
#' \item Changing the width and height with \code{width} and \code{height}. The argument \code{relavitve} determines whether relative or absolute values are used.
#' \item Setting the x and y limits. The argument \code{relavitve} determines whether relative or absolute values are used.}
#'
#' A new bounding box can be created from scratch as follows:
#' \itemize{
#' \item Using the extension factor \code{ext}.
#' \item Setting the center coorinates \code{cx} and \code{cy}, together with the \code{width} and \code{height}.
#' \item Setting the x and y limits \code{xlim} and \code{ylim}
#' }
#'
#' @param x One of the following:
#' \itemize{
#' \item A shape (from class \code{\link[sp:Spatial]{Spatial}}, \code{\link[raster:Raster-class]{Raster}}, or \code{sf} (simple features)).
#' \item A bounding box (either 2 by 2 matrix or an \code{\link[raster:Extent]{Extent}} object).
#' \item Open Street Map search query. The bounding is automatically generated by querying \code{x} from Open Street Map Nominatim. See \code{\link{geocode_OSM}} and \url{http://wiki.openstreetmap.org/wiki/Nominatim}.}
#' If \code{x} is not specified, a bounding box can be created from scratch (see details).
#' @param ext Extension factor of the bounding box. If 1, the bounding box is unchanged. Values smaller than 1 reduces the bounding box, and values larger than 1 enlarges the bounding box. This argument is a shortcut for both \code{width} and \code{height} with \code{relative=TRUE}. If a negative value is specified, then the shortest side of the bounding box (so width or height) is extended with \code{ext}, and the longest side is extended with the same absolute value. This is especially useful for bounding boxes with very low or high aspect ratios.
#' @param cx center x coordinate
#' @param cy center y coordinate
#' @param width width of the bounding box. These are either absolute or relative (depending on the argument \code{relative}).
#' @param height height of the bounding box. These are either absolute or relative (depending on the argument \code{relative}).
#' @param xlim limits of the x-axis. These are either absolute or relative (depending on the argument \code{relative}).
#' @param ylim limits of the y-axis. See \code{xlim}.
#' @param relative boolean that determines whether relative values are used for \code{width}, \code{height}, \code{xlim} and \code{ylim} or absolute. If \code{x} is unspecified, \code{relative} is set to \code{"FALSE"}.
#' @param current.projection projection that corresponds to the bounding box specified by \code{x}. See \code{\link{get_proj4}} for options.
#' @param projection projection to transform the bounding box to. See \code{\link{get_proj4}} for options.
#' @param as.extent should the bounding box be returned as \code{\link[raster:extent]{extent}}? If \code{FALSE} (default) then a matrix is returned
#' @return bounding box (see argument \code{as.extent})
#' @import sp
#' @importFrom raster extent
#' @importFrom XML xmlTreeParse xmlChildren xmlRoot xmlAttrs
#' @importFrom rgeos gIntersection
#' @example ./examples/bb.R
#' @seealso \code{\link{geocode_OSM}}
#' @export
bb <- function(x=NA, ext=NULL, cx=NULL, cy=NULL, width=NULL, height=NULL, xlim=NULL, ylim=NULL, relative = FALSE, current.projection=NULL, projection=NULL, as.extent=FALSE) {

    ## get unprocessed bounding box
    res <- get_bb(x, cx=cx, cy=cy, width=width, height=height, xlim=xlim, ylim=ylim, current.projection=current.projection)
    b <- res$b
    cx <- res$cx
    cy <- res$cy
    width <- res$width
    height <- res$height
    xlim <- res$xlim
    ylim <- res$ylim
    current.projection <- res$current.projection

    ## impute cx and cy
    if (!is.character(x)) {
		if (is.null(cx)) cx <- mean(b[1,])
		if (is.null(cy)) cy <- mean(b[2,])
	}

    ## translate ext to width and height
	steps <- b[, 2] - b[, 1]
	if (!missing(ext)) {
		relative <- TRUE
		if (ext > 0) {
			width <- ext
			height <- ext
		} else {
			if (steps[1] > steps[2]) {
				height <- -ext
				fact <- (steps[2]/ steps[1])
				if (is.nan(fact)) fact <- 1
				width <- 1 + (-ext-1) * fact
			} else {
				width <- -ext
				fact <- (steps[1]/ steps[2])
				if (is.nan(fact)) fact <- 1
				height <- 1 + (-ext-1) * fact
			}
		}
	}

	## modify bb
	if (relative) {
		xlim <- if (!is.null(xlim)) {
			b[1,1] + xlim * steps[1]
		} else if (!is.null(width)) {
			c(cx - (width/2) * steps[1],
			  cx + (width/2) * steps[1])
		} else {
			b[1, ]
		}
		ylim <- if (!is.null(ylim)) {
			b[2,1] + ylim * steps[2]
		} else if (!is.null(height)) {
			c(cy - (height/2) * steps[2],
			  cy + (height/2) * steps[2])
		} else {
			b[2, ]
		}
	} else {
		if (!is.null(width)) {
			xlim <- c(cx - (width/2),
					  cx + (width/2))
		} else if (is.null(xlim)) {
			xlim <- b[1, ]
		}
		if (!is.null(height)) {
			ylim <- c(cy - (height/2),
					  cy + (height/2))
		} else if (is.null(ylim)) {
			ylim <- b[2, ]
		}
	}

	## create bb
	b <- matrix(c(xlim, ylim), ncol = 2, byrow=TRUE,
		   dimnames=list(c("x", "y"), c("min", "max")))

	## reproject bb
	if (!missing(projection)) {
		#####################################################
		# Reproject bounding box
		#####################################################
		if (is.na(current.projection)) {
			if (!maybe_longlat(b)) {
				stop("Current projection unknown. Please specify the projection.")
			}
			warning("Current projection unknown. Long lat coordinates (wgs84) assumed.", call. = FALSE)
			current.projection <- .CRS_longlat
		} else current.projection <- get_proj4(current.projection, as.CRS = TRUE)
		projection <- get_proj4(projection, as.CRS = TRUE)

	    sp_poly <- as(extent(b), "SpatialPolygons")
	    attr(sp_poly, "proj4string") <- current.projection

	    # STEP 1: try to cut the bounding box, such that it is feasible (i.e. corresponding to lon between -180 and 180 and lat between -90 and 90)
	    earth_end <- suppressWarnings(bb_earth(projection=current.projection))

	    if (is.null(earth_end)) {
	        sp_poly2 <- sp_poly
	    } else {
	        sp_poly2 <- tryCatch({
	            gIntersection(sp_poly, earth_end)
	        }, error=function(e){
	            sp_poly
	        })
	        if (is.null(sp_poly2) || !inherits(sp_poly2, "SpatialPolygons")) sp_poly2 <- sp_poly
	    }

	    # STEP 2: Extract the bounding box corner points and add intermediate points, which can be needed since the exterme values may not be corner points once they are projected. Create a SpatialPoints objects from these points.
	    co <- sp_poly2@polygons[[1]]@Polygons[[1]]@coords
	    co2 <- apply(co, 2, function(v) {
	        n <- length(v)
	        c(v[1], rep(v[-n], each=4) + as.vector(sapply((v[-1] - v[-n]) / 4, function(w)cumsum(rep(w,4)))))
	    })
	    sp_pnts <- SpatialPoints(co2, proj4string = current.projection)

	    # STEP 3: Reproject SpatialPoints object
	    tryCatch({
	        sp_pnts2_prj <- set_projection(sp_pnts, projection=projection)
	    }, error=function(e) {
	        stop("Something went wrong with the bounding box. Please check the projection.", call.=FALSE)
	    })

	    # STEP 4: Get bounding box of reprojected object
	    b <- sp_pnts2_prj@bbox
	    dimnames(b) <- list(c("x", "y"), c("min", "max"))
	    is_prj <- is_projected(attr(projection, "projargs"))
	} else {
	    is_prj <- if (is.na(current.projection))
	        !maybe_longlat(b)
	    else is_projected(get_proj4(current.projection, as.CRS = FALSE))
	}

	## check if long lat coordinates are valid
	if (!is_prj) {
	    b[,1] <- pmax(b[,1], c(-180, -90))
	    b[,2] <- pmin(b[,2], c(180, 90))
	}
	if (as.extent) extent(b) else b
}

get_sf_bbox <- function(shp) {
    matrix(attr(shp[[attr(shp, "sf_column")]], "bbox"), ncol=2, dimnames = list(c("x", "y"), c("min", "max")))
}

get_bb <- function(x, cx=NULL, cy=NULL, width=NULL, height=NULL, xlim=NULL, ylim=NULL, current.projection=NULL) {
    if (is.character(x)) {
        res <- geocode_OSM(x)
        b <- res$bbox
        cx <- res$coords[1]
        cy <- res$coords[2]
        current.projection <- .CRS_longlat
    } else if (inherits(x, "Extent")) {
        b <- bbox(x)
    } else if (inherits(x, c("Spatial", "Raster"))) {
        b <- bbox(x)
        current.projection <- get_projection(x, as.CRS = TRUE)
    } else if (inherits(x, c("sf", "sfc"))) {
        b <- get_sf_bbox(x)
        current.projection <- get_projection(x, as.CRS = TRUE)
    } else if (is.matrix(x) && length(x)==4) {
        b <- x
    } else if (is.vector(x) && length(x)==4) {
        b <- matrix(x, ncol=2, byrow=TRUE)
    } else if (!is.na(x)[1]) {
        stop("Incorrect x argument")
    } else {
        if ((is.null(xlim) && (is.null(width) || is.null(cx))) || (is.null(xlim) && (is.null(height) || is.null(cy))))
            stop("Argument x is missing. Please specify x, or {xlim and ylim}, or {width, height, cx, and cy}.")
        ## create new bounding box
        if (is.null(xlim)) xlim <- cx + c(-.5, .5) * width
        if (is.null(ylim)) ylim <- cy + c(-.5, .5) * height
        b <- matrix(c(xlim, ylim), ncol=2,nrow=2, byrow = TRUE)
    }
    if (is.null(current.projection)) current.projection <- CRS("")
    list(b=b,
         cx=cx,
         cy=cy,
         width=width,
         height=height,
         xlim=xlim,
         ylim=ylim,
         current.projection=current.projection)
}
