% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/style_tt.R
\name{style_tt}
\alias{style_tt}
\title{Style a Tiny Table}
\usage{
style_tt(
  x,
  i = NULL,
  j = NULL,
  bold = FALSE,
  italic = FALSE,
  monospace = FALSE,
  underline = FALSE,
  strikeout = FALSE,
  color = NULL,
  background = NULL,
  fontsize = NULL,
  width = NULL,
  align = NULL,
  alignv = NULL,
  colspan = NULL,
  rowspan = NULL,
  indent = 0,
  line = NULL,
  line_color = "black",
  line_width = 0.1,
  finalize = NULL,
  tabularray_inner = NULL,
  tabularray_outer = NULL,
  bootstrap_class = NULL,
  bootstrap_css = NULL,
  bootstrap_css_rule = NULL,
  ...
)
}
\arguments{
\item{x}{A table object created by \code{tt()}.}

\item{i}{Row indices where the styling should be applied. Can be a single value or a vector. \code{i=0} is the header, and negative values are higher level headers. If \code{colspan} is used, \code{i} must be of length 1.}

\item{j}{Column indices where the styling should be applied. Can be:
\itemize{
\item Integer vectors indicating column positions.
\item Character vector indicating column names.
\item A single string specifying a Perl-style regular expression used to match column names.
}}

\item{bold}{Logical; if \code{TRUE}, text is styled in bold.}

\item{italic}{Logical; if \code{TRUE}, text is styled in italic.}

\item{monospace}{Logical; if \code{TRUE}, text is styled in monospace font.}

\item{underline}{Logical; if \code{TRUE}, text is underlined.}

\item{strikeout}{Logical; if \code{TRUE}, text has a strike through line.}

\item{color}{Text color. There are several ways to specify colors, depending on the output format.
\itemize{
\item HTML:
\itemize{
\item Hex code composed of # and 6 characters, ex: #CC79A7.
\item Keywords: black, silver, gray, white, maroon, red, purple, fuchsia, green, lime, olive, yellow, navy, blue, teal, aqua
}
\item LaTeX:
\itemize{
\item Hex code composed of # and 6 characters, ex: "#CC79A7". See the section below for instructions to add in LaTeX preambles.
\item Keywords: black, blue, brown, cyan, darkgray, gray, green, lightgray, lime, magenta, olive, orange, pink, purple, red, teal, violet, white, yellow.
\item Color blending using xcolor\verb{, ex: }white!80!blue\verb{, }green!20!red`.
\item Color names with luminance levels from \href{https://mirror.quantum5.ca/CTAN/macros/latex/contrib/ninecolors/ninecolors.pdf}{the \code{ninecolors} package} (ex: "azure4", "magenta8", "teal2", "gray1", "olive3").
}
}}

\item{background}{Background color. Specified as a color name or hexadecimal code. Can be \code{NULL} for default color.}

\item{fontsize}{Font size in em units. Can be \code{NULL} for default size.}

\item{width}{Width of column in em units. Can be \code{NULL} for default width.}

\item{align}{A single character or a string with a number of characters equal to the number of columns in \code{j}. Valid characters include 'c' (center), 'l' (left), 'r' (right), 'd' (decimal). Decimal alignment is only available in LaTeX via the \code{siunitx} package. The width of columns is determined by the maximum number of digits to the left and to the right in all cells specified by \code{i} and \code{j}.}

\item{alignv}{A single character specifying vertical alignment. Valid characters include 't' (top), 'm' (middle), 'b' (bottom).}

\item{colspan}{Number of columns a cell should span. \code{i} and \code{j} must be of length 1.}

\item{rowspan}{Number of rows a cell should span. \code{i} and \code{j} must be of length 1.}

\item{indent}{Text indentation in em units. Positive values only.}

\item{line}{String determines if solid lines (rules or borders) should be drawn around the cell, row, or column.
\itemize{
\item "t": top
\item "b": bottom
\item "l": left
\item "r": right
\item Can be combined such as: "lbt" to draw borders at the left, bottom, and top.
}}

\item{line_color}{Color of the line. See the \code{color} argument for details.}

\item{line_width}{Width of the line in em units (default: 0.1).}

\item{finalize}{A function applied to the table object at the very end of table-building, for post-processing. For example, the function could use regular expressions to add LaTeX commands to the text version of the table hosted in \code{x@table_string}, or it could programmatically change the caption in \code{x@caption}.}

\item{tabularray_inner}{A string that specifies the "inner" settings of a tabularray LaTeX table.}

\item{tabularray_outer}{A string that specifies the "outer" settings of a tabularray LaTeX table.}

\item{bootstrap_class}{String. Bootstrap table class such as \code{"table"}, \code{"table table-dark"} or \code{"table table-dark table-hover"}. See the bootstrap documentation.}

\item{bootstrap_css}{Character vector. CSS style declarations to be applied to every cell defined by \code{i} and \code{j} (ex: \code{"font-weight: bold"}).}

\item{bootstrap_css_rule}{String. Complete CSS rules (with curly braces, semicolon, etc.) that apply to the table class specified by the \code{bootstrap_class} argument.}

\item{...}{extra arguments are ignored}
}
\value{
An object of class \code{tt} representing the table.
}
\description{
Style a Tiny Table
}
\details{
This function applies styling to a table created by \code{tt()}. It allows customization of text style (bold, italic, monospace), text and background colors, font size, cell width, text alignment, column span, and indentation. The function also supports passing native instructions to LaTeX (tabularray) and HTML (bootstrap) formats.

Note: Markdown and Word tables only support these styles: italic, bold, strikeout. Moreover, the \code{style_tt()} function cannot be used to style headers inserted by the \code{group_tt()} function; instead, you should style the headers directly in the header definition using markdown syntax: \code{group_tt(i = list("*italic header*" = 2))}. These limitations are due to the fact that there is no markdown syntax for the other options, and that we create Word documents by converting a markdown table to .docx via the Pandoc software.
}
\section{LaTeX preamble}{


When rendering Quarto and Rmarkdown documents, \code{tinytable} will populate the LaTeX preamble automatically with all the required packages. For standalone LaTeX packages, these commands should be inserted in the preamble:

\if{html}{\out{<div class="sourceCode latex">}}\preformatted{\\usepackage\{tabularray\}
\\usepackage\{float\}
\\usepackage\{graphicx\}
\\usepackage[normalem]\{ulem\}
\\UseTblrLibrary\{booktabs\}
\newcommand{\tinytableTabularrayUnderline}[1]\{\\underline\{#1\}\}
\newcommand{\tinytableTabularrayStrikeout}[1]\{\\sout\{#1\}\}
\\NewTableCommand\{\\tinytableDefineColor\}[3]\{\\definecolor\{#1\}\{#2\}\{#3\}\}
}\if{html}{\out{</div>}}
}

\examples{
\dontshow{if (knitr::is_html_output()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\dontshow{\}) # examplesIf}
if (knitr::is_html_output()) options(tinytable_print_output = "html")

library(tinytable)

tt(mtcars[1:5, 1:6])

# Alignment
tt(mtcars[1:5, 1:6]) |> 
  style_tt(j = 1:5, align = "lcccr")

# Colors and styles
tt(mtcars[1:5, 1:6]) |> 
  style_tt(i = 2:3, background = "black", color = "orange", bold = TRUE)

# column selection with `j``
tt(mtcars[1:5, 1:6]) |> 
  style_tt(j = 5:6, background = "pink")

tt(mtcars[1:5, 1:6]) |>
  style_tt(j = "drat|wt", background = "pink")

tt(mtcars[1:5, 1:6]) |>
  style_tt(j = c("drat", "wt"), background = "pink")

tt(mtcars[1:5, 1:6], theme = "void") |>
  style_tt(
    i = 2, j = 2,
    colspan = 3,
    rowspan = 2,
    align="c",
    alignv = "m",
    color = "white",
    background = "black",
    bold = TRUE)
  
tt(mtcars[1:5, 1:6], theme = "void") |>
  style_tt(
    i=0:3,
    j=1:3,
    line="tblr",
    line_width=0.4,
    line_color="teal")
    
tt(mtcars[1:5, 1:6], theme = "bootstrap") |>
    style_tt(
      i = c(2,5),
      j = 3,
      strikeout = TRUE,
      fontsize = 0.7)
      
tt(mtcars[1:5, 1:6]) |>
  style_tt(bootstrap_class = "table table-dark table-hover")


inner <- "
column{1-4}={halign=c},
hlines = {fg=white},
vlines = {fg=white},
cell{1,6}{odd} = {bg=teal7},
cell{1,6}{even} = {bg=green7},
cell{2,4}{1,4} = {bg=red7},
cell{3,5}{1,4} = {bg=purple7},
cell{2}{2} = {r=4,c=2}{bg=azure7},
"
tt(mtcars[1:5, 1:4], theme = "void") |>
  style_tt(tabularray_inner = inner)

}
