% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_proportion.R
\name{h_proportions}
\alias{h_proportions}
\alias{prop_wilson}
\alias{prop_strat_wilson}
\alias{prop_clopper_pearson}
\alias{prop_wald}
\alias{prop_agresti_coull}
\alias{prop_jeffreys}
\title{Helper functions for calculating proportion confidence intervals}
\usage{
prop_wilson(rsp, conf_level, correct = FALSE)

prop_strat_wilson(
  rsp,
  strata,
  weights = NULL,
  conf_level = 0.95,
  max_iterations = NULL,
  correct = FALSE
)

prop_clopper_pearson(rsp, conf_level)

prop_wald(rsp, conf_level, correct = FALSE)

prop_agresti_coull(rsp, conf_level)

prop_jeffreys(rsp, conf_level)
}
\arguments{
\item{rsp}{(\code{logical})\cr vector indicating whether each subject is a responder or not.}

\item{conf_level}{(\code{proportion})\cr confidence level of the interval.}

\item{correct}{(\code{flag})\cr whether to apply continuity correction.}

\item{strata}{(\code{factor})\cr variable with one level per stratum and same length as \code{rsp}.}

\item{weights}{(\code{numeric} or \code{NULL})\cr weights for each level of the strata. If \code{NULL}, they are
estimated using the iterative algorithm proposed in \insertCite{Yan2010-jt;textual}{tern} that
minimizes the weighted squared length of the confidence interval.}

\item{max_iterations}{(\code{count})\cr maximum number of iterations for the iterative procedure used
to find estimates of optimal weights.}
}
\value{
Confidence interval of a proportion.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

Functions to calculate different proportion confidence intervals for use in \code{\link[=estimate_proportion]{estimate_proportion()}}.
}
\section{Functions}{
\itemize{
\item \code{prop_wilson()}: Calculates the Wilson interval by calling \code{\link[stats:prop.test]{stats::prop.test()}}.
Also referred to as Wilson score interval.

\item \code{prop_strat_wilson()}: Calculates the stratified Wilson confidence
interval for unequal proportions as described in \insertCite{Yan2010-jt;textual}{tern}

\item \code{prop_clopper_pearson()}: Calculates the Clopper-Pearson interval by calling \code{\link[stats:binom.test]{stats::binom.test()}}.
Also referred to as the \code{exact} method.

\item \code{prop_wald()}: Calculates the Wald interval by following the usual textbook definition
for a single proportion confidence interval using the normal approximation.

\item \code{prop_agresti_coull()}: Calculates the Agresti-Coull interval. Constructed (for 95\% CI) by adding two successes
and two failures to the data and then using the Wald formula to construct a CI.

\item \code{prop_jeffreys()}: Calculates the Jeffreys interval, an equal-tailed interval based on the
non-informative Jeffreys prior for a binomial proportion.

}}
\examples{
rsp <- c(
  TRUE, TRUE, TRUE, TRUE, TRUE,
  FALSE, FALSE, FALSE, FALSE, FALSE
)
prop_wilson(rsp, conf_level = 0.9)

# Stratified Wilson confidence interval with unequal probabilities

set.seed(1)
rsp <- sample(c(TRUE, FALSE), 100, TRUE)
strata_data <- data.frame(
  "f1" = sample(c("a", "b"), 100, TRUE),
  "f2" = sample(c("x", "y", "z"), 100, TRUE),
  stringsAsFactors = TRUE
)
strata <- interaction(strata_data)
n_strata <- ncol(table(rsp, strata)) # Number of strata

prop_strat_wilson(
  rsp = rsp, strata = strata,
  conf_level = 0.90
)

# Not automatic setting of weights
prop_strat_wilson(
  rsp = rsp, strata = strata,
  weights = rep(1 / n_strata, n_strata),
  conf_level = 0.90
)

prop_clopper_pearson(rsp, conf_level = .95)

prop_wald(rsp, conf_level = 0.95)
prop_wald(rsp, conf_level = 0.95, correct = TRUE)

prop_agresti_coull(rsp, conf_level = 0.95)

prop_jeffreys(rsp, conf_level = 0.95)

}
\references{
\insertRef{Yan2010-jt}{tern}
}
\seealso{
\link{estimate_proportions}, descriptive function \code{\link[=d_proportion]{d_proportion()}},
and helper functions \code{\link[=strata_normal_quantile]{strata_normal_quantile()}} and \code{\link[=update_weights_strat_wilson]{update_weights_strat_wilson()}}.
}
