#' Get accepted names from tsn
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getacceptednamesfromtsn('208527')  # TSN accepted - good name
#' getacceptednamesfromtsn(tsn='504239')  # TSN not accepted - input TSN is old name
#' getacceptednamesfromtsn('504239', FALSE)  # TSN not accepted - input TSN is old name
#' }
getacceptednamesfromtsn <- function(tsn = NA, ..., curl = getCurlHandle())
{ 
	url = "http://www.itis.gov/ITISWebService/services/ITISService/getAcceptedNamesFromTSN"
	args <- list()
	if(!is.na(tsn))
		args$tsn <- tsn
	tt <- getForm(url, 
								.params = args, 
								...,
								curl = curl)
	tt_ <- xmlParse(tt)
	temp <- xmlToList(tt_)
	if(length(temp$return$acceptedNames) == 1) 
	{ message("Good name!")
		temp$return$tsn
	} else
	{ message("Bad name!")
		c(submittedTsn = temp$return$tsn, temp$return$acceptedNames[1:2])
	}
}

#' Get any match count.
#' 
#' @import XML RCurl
#' @param srchkey text or taxonomic serial number (TSN) (character or numeric)
#' @param ... optional additional curl options (debugging tools mostly)
#' @param curl If using in a loop, call getCurlHandle() first and pass 
#'  the returned value in here (avoids unnecessary footprint)
#' @return An integer containing the number of matches the search will return.
#' @examples \dontrun{
#' getanymatchcount(202385)
#' getanymatchcount("dolphin")
#' }
#' @export 
getanymatchcount <- function(srchkey = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getAnyMatchCount'
  args <- list()
  if(!is.na(srchkey))
    args$srchKey <- srchkey
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  xmlToList(out)
}

#' Get comment detail from TSN
#' 
#' @import XML RCurl
#' @param tsn TSN for a taxonomic group (numeric)
#' @param ... optional additional curl options (debugging tools mostly)
#' @param curl If using in a loop, call getCurlHandle() first and pass 
#'  the returned value in here (avoids unnecessary footprint)
#' @return A data.frame with results.
#' @examples \dontrun{
#' getcommentdetailfromtsn(180543)
#' }
#' @export
getcommentdetailfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getCommentDetailFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  nodes <- getNodeSet(out, "//ax21:commentDetail", namespaces=namespaces)
  comment <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:commentId", namespaces=namespaces)
  commid <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:commentTimeStamp", namespaces=namespaces)
  commTime <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:commentator", namespaces=namespaces)
  commentator <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:updateDate", namespaces=namespaces)
  updatedate <- sapply(nodes, xmlValue)
  data.frame(comment=comment, commid=commid, commTime=commTime, 
    commentator=commentator, updatedate=updatedate)
}

#' Get common names from tsn
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getcommonnamesfromtsn(183833)
#' }
#' @export 
getcommonnamesfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getCommonNamesFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  nodes <- getNodeSet(out, "//ax21:commonName", namespaces=namespaces)
  comname <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:language", namespaces=namespaces)
  lang <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue)
  data.frame(comname=comname, lang=lang, tsn=tsn[-length(tsn)])
}


#' Get core metadata from tsn
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getcoremetadatafromtsn(tsn = 28727)  # coverage and currrency data
#' getcoremetadatafromtsn(tsn = 183671)  # no coverage or currrency data
#' }
#' @export 
getcoremetadatafromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getCoreMetadataFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("credRating","rankId","taxonCoverage","taxonCurrency","taxonUsageRating","tsn")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Get coverge from tsn
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getcoveragefromtsn(tsn = 28727)  # coverage data
#' getcoveragefromtsn(tsn = 526852)  # no coverage data
#' }
#' @export 
getcoveragefromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getCoverageFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  nodes <- getNodeSet(out, "//ax21:rankId", namespaces=namespaces)
  rankid <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:taxonCoverage", namespaces=namespaces)
  taxoncoverage <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue)
  data.frame(rankid=rankid, taxoncoverage=taxoncoverage, tsn=tsn)
}

#' Get credibility rating from tsn
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getcredibilityratingfromtsn(tsn = 526852)
#' }
#' @export 
getcredibilityratingfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getCredibilityRatingFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  nodes <- getNodeSet(out, "//ax21:credRating", namespaces=namespaces)
  credrating <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue)
  data.frame(credrating=credrating, tsn=tsn)
}

#' Get possible credibility ratings
#' 
#' @import RCurl XML
#' @examples \dontrun{
#' getcredibilityratings()
#' }
#' @export 
getcredibilityratings <- function() 
{
	url='http://www.itis.gov/ITISWebService/services/ITISService/getCredibilityRatings'
  message(url)
  tt <- getURL(url)
  out <- xmlParse(tt)
  namespaces <- c(ax25="http://metadata.itis_service.itis.usgs.org/xsd")
  nodes <- getNodeSet(out, "//ax25:credibilityValues", namespaces=namespaces)
  credibilityValues <- sapply(nodes, xmlValue)
  data.frame(credibilityValues = credibilityValues)
}

#' Get currency from tsn
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getcurrencyfromtsn(tsn = 28727) # currency data
#' getcurrencyfromtsn(tsn = 526852) # no currency dat
#' }
#' @export 
getcurrencyfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getCurrencyFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(ax21="http://data.itis_service.itis.usgs.org/xsd")
  nodes <- getNodeSet(out, "//ax21:rankId", namespaces=namespaces)
  rankid <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:taxonCurrency", namespaces=namespaces)
  taxoncurrency <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue)
  data.frame(rankid=rankid, taxoncurrency=taxoncurrency, tsn=tsn)
}

#' Get date data from tsn
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getdatedatafromtsn(tsn = 180543)
#' }
#' @export 
getdatedatafromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getDateDataFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(ax21="http://data.itis_service.itis.usgs.org/xsd")
  nodes <- getNodeSet(out, "//ax21:initialTimeStamp", namespaces=namespaces)
  initialTimeStamp <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:updateDate", namespaces=namespaces)
  updateDate <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue)
  data.frame(initialTimeStamp=initialTimeStamp, updateDate=updateDate, tsn=tsn)
}

#' Get description of the ITIS service
#' 
#' @import RCurl XML
#' @examples \dontrun{
#' getdescription()
#' }
#' @export 
getdescription <- function() 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getDescription'
  message(url)
  tt <- getURL(url)
  out <- xmlParse(tt)
  namespaces <- c(ax26="http://itis_service.itis.usgs.org/xsd")
  nodes <- getNodeSet(out, "//ax26:description", namespaces=namespaces)
  sapply(nodes, xmlValue)
}

#' Get expert information for the TSN.
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getexpertsfromtsn(tsn = 180544)
#' }
#' @export 
getexpertsfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getExpertsFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("comment","expert","name","referredTsn","referenceFor","updateDate")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Get full hierarchy from tsn
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getfullhierarchyfromtsn(tsn = 37906)
#' getfullhierarchyfromtsn(tsn = 500664)
#' }
#' @export 
getfullhierarchyfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getFullHierarchyFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  nodes <- getNodeSet(out, "//ax21:parentName", namespaces=namespaces)
  parentName <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:parentTsn", namespaces=namespaces)
  parentTsn <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:rankName", namespaces=namespaces)
  rankName <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:taxonName", namespaces=namespaces)
  taxonName <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue)
  out <- data.frame(parentName=parentName, parentTsn=parentTsn, rankName=rankName,
             taxonName=taxonName, tsn=tsn[-1])
  return(out)
}

#' Returns the full ITIS record for the TSN in the LSID, found by comparing the 
#' 		TSN in the search key to the TSN field. Returns an empty result set if there 
#'  	is no match or the TSN is invalid.
#' 
#' @import RCurl XML
#' @param lsid lsid for a taxonomic group (character)
#' @param ... optional additional curl options (debugging tools mostly)
#' @param curl If using in a loop, call getCurlHandle() first and pass 
#'  the returned value in here (avoids unnecessary footprint) 
#' @examples \dontrun{
#' getfullrecordfromlsid(lsid = "urn:lsid:itis.gov:itis_tsn:180543")
#' }
#' @export 
getfullrecordfromlsid <- function(lsid = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getFullRecordFromLSID'
  args <- list()
  if(!is.na(lsid))
    args$lsid <- lsid
  message(paste(url, '?lsid=', lsid, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  xmlParse(tt)
}

#' Get full record from TSN.
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getfullrecordfromtsn(tsn = 183833)
#' }
#' @export 
getfullrecordfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getFullRecordFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  xmlParse(tt)
}

#' Get geographic divisions from tsn
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getgeographicdivisionsfromtsn(tsn = 180543)
#' }
#' @export 
getgeographicdivisionsfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getGeographicDivisionsFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("geographicValue","updateDate","tsn")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Get all possible geographic values
#' 
#' @import RCurl XML
#' @examples \dontrun{
#' getgeographicvalues()
#' }
#' @export 
getgeographicvalues <- function() 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getGeographicValues'
  message(url)
  tt <- getURL(url)
  out <- xmlParse(tt)
  namespaces <- c(ax21="http://metadata.itis_service.itis.usgs.org/xsd" )
  nodes <- getNodeSet(out, "//ax21:geographicValues", namespaces=namespaces)
  geographicValues <- sapply(nodes, xmlValue)
  data.frame(geographicValues = geographicValues)
}

#' Get global species completeness from tsn
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getglobalspeciescompletenessfromtsn(tsn = 180541)
#' }
#' @export 
getglobalspeciescompletenessfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getGlobalSpeciesCompletenessFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("completeness","rankId","tsn")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Get hierarchy down from tsn
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' gethierarchydownfromtsn(tsn = 161030)
#' }
#' @export 
gethierarchydownfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getHierarchyDownFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  nodes <- getNodeSet(out, "//ax21:parentName", namespaces=namespaces)
  parentName <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:parentTsn", namespaces=namespaces)
  parentTsn <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:rankName", namespaces=namespaces)
  rankName <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:taxonName", namespaces=namespaces)
  taxonName <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue)
  data.frame(parentName=parentName, parentTsn=parentTsn, rankName=rankName,
             taxonName=taxonName, tsn=tsn[-1])
}

#' Get hierarchy up from tsn
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' gethierarchyupfromtsn(tsn = 36485)
#' }
#' @export 
gethierarchyupfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getHierarchyUpFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  nodes <- getNodeSet(out, "//ax21:parentName", namespaces=namespaces)
  parentName <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:parentTsn", namespaces=namespaces)
  parentTsn <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:rankName", namespaces=namespaces)
  rankName <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:taxonName", namespaces=namespaces)
  taxonName <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue)
  data.frame(parentName=parentName, parentTsn=parentTsn, rankName=rankName,
             taxonName=taxonName, tsn=tsn)
}

#' Get itis terms from common names
#' 
#' @inheritParams getanymatchcount
#' @examples \dontrun{
#' getitistermsfromcommonname("buya")
#' }
#' @export 
getitistermsfromcommonname <- function(srchkey = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getITISTermsFromCommonName'
  args <- list()
  if(!is.na(srchkey))
    args$srchKey <- srchkey
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  nodes <- getNodeSet(out, "//ax21:commonNames", namespaces=namespaces)
  comname <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:nameUsage", namespaces=namespaces)
  nameusage <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:scientificName", namespaces=namespaces)
  sciname <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue)
  data.frame(comname=comname[-length(comname)], nameusage=nameusage, sciname=sciname, tsn=tsn)
}

#' Get itis terms from scientific names
#' 
#' @inheritParams getanymatchcount
#' @examples \dontrun{
#' getitistermsfromscientificname(srchkey = "ursidae")
#' }
#' @export 
getitistermsfromscientificname <- function(srchkey = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getITISTermsFromScientificName'
  args <- list()
  if(!is.na(srchkey))
    args$srchKey <- srchkey
  message(paste(url, '?srchKey=', srchkey, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  nodes <- getNodeSet(out, "//ax21:commonNames", namespaces=namespaces)
  comname <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:nameUsage", namespaces=namespaces)
  nameusage <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:scientificName", namespaces=namespaces)
  sciname <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue)
  data.frame(comname=comname, nameusage=nameusage, sciname=sciname, tsn=tsn)
}

#' Get jurisdictional origin from tsn
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getjurisdictionaloriginfromtsn(tsn = 180543)
#' }
#' @export 
getjurisdictionaloriginfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getJurisdictionalOriginFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("jurisdictionValue","origin","updateDate")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Get jurisdiction origin values
#' 
#' @import RCurl XML
#' @examples \dontrun{
#' getjurisdictionoriginvalues()
#' }
#' @export 
getjurisdictionoriginvalues <- function() 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getJurisdictionalOriginValues'
  message(url)
  tt <- getURL(url)
  out <- xmlParse(tt)
  namespaces <- c(ax21="http://metadata.itis_service.itis.usgs.org/xsd" )
  nodes <- getNodeSet(out, "//ax21:jurisdiction", namespaces=namespaces)
  jurisdiction <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:origin", namespaces=namespaces)
  origin <- sapply(nodes, xmlValue)
  data.frame(jurisdiction = jurisdiction, origin = origin)
}

#' Get possible jurisdiction values
#' 
#' @import RCurl XML
#' @examples \dontrun{
#' getjurisdictionvalues()
#' }
#' @export 
getjurisdictionvalues <- function() 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getJurisdictionValues'
  message(url)
  tt <- getURL(url)
  out <- xmlParse(tt)
  namespaces <- c(ax21="http://metadata.itis_service.itis.usgs.org/xsd" )
  nodes <- getNodeSet(out, "//ax21:jurisdictionValues", namespaces=namespaces)
  jurisdictionValues <- sapply(nodes, xmlValue)
  data.frame(jurisdictionValues = jurisdictionValues)
}

#' Get kingdom names from tsn
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getkingdomnamefromtsn(tsn = 202385)
#' }
#' @export 
getkingdomnamefromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getKingdomNameFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("kingdomId","kingdomName","tsn")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Get all possible kingdom names
#' 
#' @import RCurl XML
#' @examples \dontrun{
#' getkingdomnamefromtsn(tsn = 202385)
#' }
#' @export 
getkingdomnames <- function() 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getKingdomNames'
  message(url)
  tt <- getURL(url)
  out <- xmlParse(tt)
  namespaces <- c(ax21="http://metadata.itis_service.itis.usgs.org/xsd")
  nodes <- getNodeSet(out, "//ax21:kingdomId", namespaces=namespaces)
  kingdomId <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:kingdomName", namespaces=namespaces)
  kingdomName <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue)
  data.frame(kingdomId = kingdomId, kingdomName = kingdomName, tsn = tsn)
}

#' Provides the date the ITIS database was last updated.
#' 
#' @import RCurl XML
#' @examples \dontrun{
#' getlastchangedate()
#' }
#' @export 
getlastchangedate <- function() 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getLastChangeDate'
  message(url)
  tt <- getURL(url)
  out <- xmlParse(tt)
  namespaces <- c(ax21="http://metadata.itis_service.itis.usgs.org/xsd")
  nodes <- getNodeSet(out, "//ax21:updateDate", namespaces=namespaces)
  sapply(nodes, xmlValue)
}

#' Gets the unique LSID for the TSN, or an empty result if there is no match.
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getlsidfromtsn(tsn = 155166)
#' }
#' @export 
getlsidfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getLSIDFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
#     ...,
    curl = curl)
  xmlToList(xmlParse(tt))[[1]]
}

#' Returns a list of the other sources used for the TSN.
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getothersourcesfromtsn(tsn = 182662)
#' }
#' @export 
getothersourcesfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getOtherSourcesFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("acquisitionDate","name","referredTsn","source",
                "sourceType","updateDate","version")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Returns the parent TSN for the entered TSN.
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getparenttsnfromtsn(tsn = 202385)
#' }
#' @export 
getparenttsnfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getParentTSNFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("parentTsn","tsn")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Returns a list of the pulications used for the TSN.
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getpublicationsfromtsn(tsn = 70340)
#' }
#' @export 
getpublicationsfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getPublicationsFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("actualPubDate","isbn","issn","listedPubDate","pages",
                "pubComment","pubName","pubPlace","publisher","referenceAuthor",
                "name","refLanguage","referredTsn","title","updateDate")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Provides a list of all the unique rank names contained in the database and 
#'  their kingdom and rank ID values.
#' 
#' @import RCurl XML
#' @examples \dontrun{
#' getranknames()
#' }
#' @export 
getranknames <- function() 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getRankNames'
  message(url)
  tt <- getURL(url)
  out <- xmlParse(tt)
  namespaces <- c(ax25="http://metadata.itis_service.itis.usgs.org/xsd")
  nodes <- getNodeSet(out, "//ax25:kingdomName", namespaces=namespaces)
  kingdomName <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax25:rankId", namespaces=namespaces)
  rankId <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax25:rankName", namespaces=namespaces)
  rankName <- sapply(nodes, xmlValue)
  data.frame(kingdomName = kingdomName, rankId = rankId, rankName = rankName)
}

#' Gets the partial ITIS record for the TSN in the LSID, found by comparing the 
#'  TSN in the search key to the TSN field. Returns an empty result set if 
#'  there is no match or the TSN is invalid.
#' 
#' @import RCurl XML
#' @param lsid lsid for a taxonomic group (numeric)
#' @param ... optional additional curl options (debugging tools mostly)
#' @param curl If using in a loop, call getCurlHandle() first and pass 
#'  the returned value in here (avoids unnecessary footprint)
#' @examples \dontrun{
#' getrecordfromlsid(lsid = "urn:lsid:itis.gov:itis_tsn:180543")
#' }
#' @export 
getrecordfromlsid <- function(lsid = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getRecordFromLSID'
  args <- list()
  if(!is.na(lsid))
    args$lsid <- lsid
  message(paste(url, '?lsid=', lsid, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("authorship","genusPart","infragenericEpithet",
                "infraspecificEpithet","lsid","nameComplete","nomenclaturalCode",
                "rank","rankString","specificEpithet","uninomial","tsn")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Returns the review year for the TSN.
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getreviewyearfromtsn(tsn = 180541)
#' }
#' @export 
getreviewyearfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getReviewYearFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("rankId","reviewYear","tsn")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Returns the scientific name for the TSN. Also returns the component parts 
#'    (names and indicators) of the scientific name.
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getscientificnamefromtsn(tsn = 531894)
#' }
#' @export 
getscientificnamefromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getScientificNameFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("combinedName","unitInd1","unitInd3","unitName1","unitName2",
                "unitName3","tsn")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Returns a list of the synonyms (if any) for the TSN.
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getsynonymnamesfromtsn(tsn = 183671) # tsn not accepted
#' getsynonymnamesfromtsn(tsn = 526852) # tsn accepted
#' }
#' @export 
getsynonymnamesfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getSynonymNamesFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
#     ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd")
  nodes <- getNodeSet(out, "//ax21:name", namespaces=namespaces)
  if( length(sapply(nodes, xmlValue)) == 0){ name <- list("nomatch") } else
    { name <- sapply(nodes, xmlValue) }
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  if( length(sapply(nodes, xmlValue)) == 1){ tsn <- sapply(nodes, xmlValue) } else
    { tsn <- sapply(nodes, xmlValue) 
      tsn <- tsn[-length(tsn)]
    } 
  data.frame(name=name[[1]], tsn=tsn)
}

#' Returns the author information for the TSN.
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' gettaxonauthorshipfromtsn(tsn = 183671)
#' }
#' @export 
gettaxonauthorshipfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getTaxonAuthorshipFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("authorship","updateDate","tsn")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Returns the kingdom and rank information for the TSN.
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' gettaxonomicranknamefromtsn(tsn = 202385)
#' }
#' @export 
gettaxonomicranknamefromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getTaxonomicRankNameFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
     ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("kingdomId","kingdomName","rankId","rankName","tsn")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Returns the usage information for the TSN.
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' gettaxonomicusagefromtsn(tsn = 526852)
#' }
#' @export 
gettaxonomicusagefromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getTaxonomicUsageFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("taxonUsageRating","tsn")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Retrieve accepted TSN (with accepted name)
#' 
#' @import RCurl XML
#' @param language A string containing the language. This is a language string, 
#'    not the international language code (character)
#' @param ... optional additional curl options (debugging tools mostly)
#' @param curl If using in a loop, call getCurlHandle() first and pass 
#'  the returned value in here (avoids unnecessary footprint)
#' @examples \dontrun{
#' gettsnbyvernacularlanguage("french")
#' }
#' @export 
gettsnbyvernacularlanguage <- function(language = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getTsnByVernacularLanguage'
  args <- list()
  if(!is.na(language))
    args$language <- language
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  nodes <- getNodeSet(out, "//ax21:commonName", namespaces=namespaces)
  comname <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:language", namespaces=namespaces)
  language <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue)
  data.frame(comname=comname, language=language, tsn=tsn)
}

#' Gets the TSN corresponding to the LSID, or an empty result if there is no match.
#' 
#' @param lsid lsid for a taxonomic group (numeric)
#' @param ... optional additional curl options (debugging tools mostly)
#' @param curl If using in a loop, call getCurlHandle() first and pass 
#'  the returned value in here (avoids unnecessary footprint)
#' @examples \dontrun{
#' gettsnfromlsid(lsid = "urn:lsid:itis.gov:itis_tsn:28726")
#' gettsnfromlsid(lsid = "urn:lsid:itis.gov:itis_tsn:0")
#' }
#' @export 
gettsnfromlsid <- function(lsid = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getTSNFromLSID'
  args <- list()
  if(!is.na(lsid))
    args$lsid <- lsid
  message(paste(url, '?lsid=', lsid, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  if( !is.na( suppressWarnings(as.numeric(xmlToList(out)[[1]])) ) )
    { suppressWarnings( as.numeric(xmlToList(out)[[1]]) )} else
      {"invalid TSN"}
}

#' Returns the unacceptability reason, if any, for the TSN.
#' 
#' @inheritParams getcommentdetailfromtsn
#' @examples \dontrun{
#' getunacceptabilityreasonfromtsn(tsn = 183671)
#' }
#' @export 
getunacceptabilityreasonfromtsn <- function(tsn = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getUnacceptabilityReasonFromTSN'
  args <- list()
  if(!is.na(tsn))
    args$tsn <- tsn
  message(paste(url, '?tsn=', tsn, sep=''))
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  toget <- list("tsn","unacceptReason")
  xpathfunc <- function(x) {    
    sapply(getNodeSet(out, paste("//ax21:", x, sep=''), namespaces=namespaces),xmlValue)
  }
  df <-  do.call(cbind, lapply(toget, as.data.frame(xpathfunc)))
  names(df) <- toget
  df
}

#' Provides a list of the unique languages used in the vernacular table.
#' 
#' @import RCurl XML
#' @examples \dontrun{
#' getvernacularlanguages()
#' }
#' @export 
getvernacularlanguages <- function() 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/getVernacularLanguages'
  message(url)
  tt <- getURL(url)
  out <- xmlParse(tt)
  namespaces <- c(ax21="http://metadata.itis_service.itis.usgs.org/xsd")
  nodes <- getNodeSet(out, "//ax21:languageNames", namespaces=namespaces)
  languageNames <- sapply(nodes, xmlValue)
  data.frame(languageNames = languageNames)
}

#' Search for tsn by common name
#' 
#' @inheritParams getanymatchcount
#' @examples \dontrun{
#' searchbycommonname("american bullfrog")
#' searchbycommonname("ferret-badger")
#' searchbycommonname(srchkey="polar bear")
#' }
#' @export 
searchbycommonname <- function(srchkey = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/searchByCommonName'
  args <- list()
  if(!is.na(srchkey))
    args$srchKey <- srchkey
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  nodes <- getNodeSet(out, "//ax21:commonName", namespaces=namespaces)
  comname <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:language", namespaces=namespaces)
  lang <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue)
  data.frame(comname=comname, lang=lang, tsn=tsn[-1])
}

#' Search for tsn by common name beginning with 
#' 
#' @inheritParams getanymatchcount
#' @examples \dontrun{
#' searchbycommonnamebeginswith("inch")
#' }
#' @export 
searchbycommonnamebeginswith <- function(srchkey = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/searchByCommonNameBeginsWith'
  args <- list()
  if(!is.na(srchkey))
    args$srchKey <- srchkey
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  nodes <- getNodeSet(out, "//ax21:commonName", namespaces=namespaces)
  comname <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:language", namespaces=namespaces)
  lang <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue) # last one is a repeat
  nodes <- getNodeSet(out, "//ax21:sciName", namespaces=namespaces)
  data.frame(comname=comname, lang=lang, tsn=tsn[-length(tsn)])
}

#' Search for tsn by common name ending with
#' 
#' @inheritParams getanymatchcount
#' @examples \dontrun{
#' searchbycommonnameendswith(srchkey="snake")
#' }
#' @export 
searchbycommonnameendswith <- function(srchkey = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/searchByCommonNameEndsWith'
  args <- list()
  if(!is.na(srchkey))
    args$srchKey <- srchkey
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  nodes <- getNodeSet(out, "//ax21:commonName", namespaces=namespaces)
  comname <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:language", namespaces=namespaces)
  lang <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue) # last one is a repeat
  data.frame(comname=comname, lang=lang, tsn=tsn[!nchar(tsn) == 0])
}

#' Search by scientific name
#' 
#' @inheritParams getanymatchcount
#' @examples \dontrun{
#' searchbyscientificname("Tardigrada")
#' searchbyscientificname("Quercus douglasii")
#' }
#' @export 
searchbyscientificname <- function(srchkey = NA, ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/searchByScientificName'
  args <- list()
  if(!is.na(srchkey))
    args$srchKey <- srchkey
  tt <- getForm(url,
    .params = args,
#     ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd")
  nodes <- getNodeSet(out, "//ax21:combinedName", namespaces=namespaces)
  combinedname <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue) # last one is a repeat
  data.frame(combinedname=combinedname, tsn=tsn)
}

#' Search for any match
#' 
#' @inheritParams getanymatchcount
#' @examples \dontrun{
#' searchforanymatch(srchkey = 202385)
#' searchforanymatch(srchkey = "dolphin")
#' }
#' @export 
searchforanymatch <- function(srchkey = NA,  ..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/searchForAnyMatch'
  args <- list()
  if(!is.na(srchkey))
    args$srchKey <- srchkey
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd")
  nodes <- getNodeSet(out, "//ax21:commonName", namespaces=namespaces)
  comname <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:language", namespaces=namespaces)
  lang <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue) # last one is a repeat
  nodes <- getNodeSet(out, "//ax21:sciName", namespaces=namespaces)
  sciName <- sapply(nodes, xmlValue)
  list(comname=comname, lang=lang, tsn=tsn[-length(tsn)], sciName=sciName)
}

#' Search for any matched page
#' 
#' @inheritParams getanymatchcount
#' @param pagesize An integer containing the page size (numeric)
#' @param pagenum An integer containing the page number (numeric)
#' @param ascend A boolean containing true for ascending sort order or false for descending (logical)
#' @examples \dontrun{
#' searchforanymatchpaged(202385, 100, 1, FALSE)
#' }
#' @export 
searchforanymatchpaged <- function(srchkey = NA, pagesize = NA, pagenum = NA, ascend = NA,
 	..., curl = getCurlHandle() ) 
{
	url = 'http://www.itis.gov/ITISWebService/services/ITISService/searchForAnyMatchPaged'
  args <- list()
  if(!is.na(srchkey))
    args$srchKey <- srchkey
  if(!is.na(pagesize))
    args$pageSize <- pagesize
  if(!is.na(pagenum))
    args$pageNum <- pagenum
  if(!is.na(ascend))
    args$ascend <- ascend
  tt <- getForm(url,
    .params = args,
    ...,
    curl = curl)
  out <- xmlParse(tt)
  namespaces <- c(namespaces <- c(ax21="http://data.itis_service.itis.usgs.gov/xsd"))
  nodes <- getNodeSet(out, "//ax21:commonName", namespaces=namespaces)
  comname <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:language", namespaces=namespaces)
  lang <- sapply(nodes, xmlValue)
  nodes <- getNodeSet(out, "//ax21:tsn", namespaces=namespaces)
  tsn <- sapply(nodes, xmlValue) # last one is a repeat
  nodes <- getNodeSet(out, "//ax21:sciName", namespaces=namespaces)
  sciName <- sapply(nodes, xmlValue)
  list(comname=comname, lang=lang, tsn=tsn[-length(tsn)], sciName=sciName)
}