#' @title Cue to force a target to run if a condition is true
#' @export
#' @family cues
#' @description `tar_cue_force()` creates a cue object to
#'   force a target to run if an arbitrary condition evaluates to `TRUE`.
#'   Supply the returned cue object to the `cue` argument of
#'   `targets::tar_target()` or similar.
#' @details `tar_cue_force()` and [tar_force()] operate differently.
#'   The former defines a cue object based on an eagerly evaluated
#'   condition, and [tar_force()] puts the condition in a special
#'   upstream target that always runs. Unlike `tar_cue_force()`,
#'   the condition in [tar_force()] can depend on upstream targets,
#'   but the drawback is that targets defined with [tar_force()]
#'   will always show up as outdated in functions like `tar_outdated()`
#'   and `tar_visnetwork()` even though `tar_make()` may still
#'   skip the main target if the condition is not met.
#' @return A cue object. See the "Cue objects" section for background.
#' @section Cue objects:
#'   A cue object is an object generated by `targets::tar_cue()`,
#'   `tarchetypes::tar_cue_force()`, or similar. It is
#'   a collection of decision rules that decide when a target
#'   is invalidated/outdated (e.g. when `tar_make()` or similar
#'   reruns the target). You can supply these cue objects to the
#'   `tar_target()` function or similar. For example,
#'   `tar_target(x, run_stuff(), cue = tar_cue(mode = "always"))`
#'   is a target that always calls `run_stuff()` during `tar_make()`
#'   and always shows as invalidated/outdated in `tar_outdated()`,
#'   `tar_visnetwork()`, and similar functions.
#' @inheritParams targets::tar_cue
#' @param condition Logical vector evaluated locally when the target is
#'   defined. If any element of `condition` is `TRUE`, the target will
#'   definitely rerun when the pipeline runs.
#'   Otherwise, the target may or may not rerun, depending
#'   on the other invalidation rules. `condition` is evaluated
#'   when this cue factory is called, so the condition cannot
#'   depend on upstream targets, and it should be quick to calculate.
#' @examples
#' if (identical(Sys.getenv("TAR_LONG_EXAMPLES"), "true")) {
#' targets::tar_dir({ # tar_dir() runs code from a temporary directory.
#' targets::tar_script({
#'   library(tarchetypes)
#'   list(
#'     targets::tar_target(
#'       data,
#'       data.frame(x = seq_len(26)),
#'       cue = tarchetypes::tar_cue_force(1 > 0)
#'     )
#'   )
#' })
#' targets::tar_make()
#' targets::tar_make()
#' })
#' }
tar_cue_force <- function(
  condition,
  command = TRUE,
  depend = TRUE,
  format = TRUE,
  iteration = TRUE,
  file = TRUE
) {
  mode <- if_any(as.logical(condition), "always", "thorough")
  targets::tar_cue(
    mode = mode,
    command = command,
    depend = depend,
    format = format,
    iteration = iteration,
    file = file
  )
}
