#' Create a musical phrase from string/fret combinations
#'
#' Create a musical phrase from character strings that define string numbers,
#' fret numbers and note metadata. This function is a wrapper around
#' \code{\link{phrase}}.
#' It allows for specifying string/fret combinations instead of unambiguous
#' pitch as is used by \code{phrase}.
#' In order to remove ambiguity, it is critical to specify the instrument
#' string tuning and key signature.
#' It essentially uses \code{string} and \code{fret} in combination with a
#' known tuning and key signature to generate \code{notes} for
#' \code{\link{phrase}}.
#' \code{info} is passed straight through to \code{phrase}, as is \code{string}
#' once it is done being used to help derive \code{notes}.
#'
#' @details
#' See the main function \code{phrase} for general details on phrase
#' construction.
#'
#' @section Comparison with \code{phrase}:
#' This function is a wrapper function for users not working with musical notes
#' (what to play), but rather just position on the guitar neck (where to play).
#' This approach has conveniences, but is more limiting.
#' In order to remove ambiguity, it is necessary to specify the instrument
#' \code{tuning} and the \code{key} signature.
#'
#' In the standard approach with \code{phrase} you specify what to play;
#' specifying exactly where to play is optional, but highly recommended (by
#' providing \code{string}).
#' With \code{sf_phrase}, the \code{string} argument is of course required
#' along with \code{fret}.
#' But any time the tuning changes, this "where to play" method breaks down and
#' must be redone. It is more robust to provide the string and pitch rather
#' than the string and fret.
#' The \code{key} is additionally important because it is the only way to
#' indicate if accidentals should be notated as sharps or flats.
#'
#' This wrapper also increases redundancy and typing. In order to specify rests
#' \code{r}, silent rests \code{s}, and tied notes \code{~}, these must now be
#' providing in parallel in both the \code{string} and \code{fret} arguments,
#' whereas in the standard method using \code{phrase}, they need only be
#' provided once to \code{notes}.
#' A mismatch will throw an error. Despite the redundancy, this is helpful for
#' ensuring proper match up between \code{string} and \code{fret},
#' which is essentially a dual entry method that aims to reduce itself inside
#' \code{sf_phrase} to a single \code{notes} string that is passed internally
#' to \code{phrase}.
#'
#' The important thing to keep in mind is that by its nature, this method of
#' writing out music does not lend itself well to high detail.
#' Tabs that are informed by nothing but string and fret number remove a lot of
#' important information, and those that attempt to compensate with additional
#' symbols in say, an ascii tab, are difficult to read.
#' This wrapper function providing this alternative input method to
#' \code{phrase} does its job of allowing users to create phrase objects that
#' are equivalent to standard \code{phrase}-generated objects, including rests
#' and ties.
#' But practice and comfort working with \code{phrase} is is highly recommended
#' for greater control of development support.
#'
#' The function \code{sfp} is a convenient shorthand wrapper for
#' \code{sf_phrase}. \code{sf_note} and the alias \code{sfn} are wrappers
#' around \code{sf_phrase} that force \code{to_notes = TRUE}.
#'
#' @section Single-string input:
#' Another way to use \code{sf_phrase} is to provide all musical input to
#' \code{string} and ignore \code{fret} and \code{info} as explicit arguments.
#' Providing all three explicit arguments more closely mimics the inputs of
#' \code{phrase} and is useful when you have this information as three
#' independent sources.
#' However, in some cases the single-argument input method can reduce typing,
#' though this depends on the phrase.
#' More importantly, it allow you to reason about your musical inputs by time
#' step rather than by argument.
#' If you provide all three components as a single character string to the
#' \code{string} argument, leaving both \code{fret} and \code{info} as
#' \code{NULL}, then \code{sf_phrase} will decompose \code{string} into its
#' three component parts internally.
#'
#' There are some rules for single-argument input. The three components are
#' separated by semicolons as \code{"string;fret;info"}.
#' For example, \code{"3;7x7;4"} means begin on the third string
#' (infer higher number strings muted).
#' The frets are 7th and 7th, meaning two notes are played. When an \code{x} is
#' present in the second entry it means a string is not played.
#' This is how it is inferred that the string numbers starting from the third
#' string are strings 3 and 1 rather than 3 and 2 in this example.
#' The 4 indicates a quarter note since it is part of the third entry where the
#' additional \code{info} is specified. This is contextual. For example, an
#' \code{x} here would still indicate a dead note, rather than an unplayed
#' string in the second entry, so this is contextual.
#'
#' A bonus when using this input method is that explicit \code{string} and
#' \code{info} values persist from one timestep to the next.
#' Neither needs to be provided again until there is a change in value.
#' For example, \code{"3;7x7;4 7x7 ;7x7;1"} repeats the string and info values
#' from timestep one for timestep two.
#' In timestep three, string numbers repeat again, but the duration changes
#' from quarter note to whole note.
#'
#' Note that except when both \code{string} and \code{info} are repeating and
#' only fret numbers are provided (see timestep two above), two semicolons
#' must be present so that it is unambiguous whether the sole missing component
#' is a \code{string} or \code{info} (see timestep three).
#'
#' Ambiguity would arise from a case like \code{"4;4"} without the second
#' semicolon. This type of indexing was chosen over using two different
#' delimiters.
#'
#' If a rest, \code{r} or \code{s}, is provided for the \code{fret} entry,
#' then the \code{string} entry is ignored.
#' When using this input method, ties \code{~} are given in the info entry.
#'
#' See the examples for a comparison of two identical phrases specified using
#' both input methods for \code{sf_phrase}.
#'
#' @param string character, string numbers associated with notes, or provide
#' all information here and ignore \code{fret} and \code{info}. See details.
#' @param fret character, fret numbers associated with notes.
#' @param info character, metadata associated with notes.
#' @param key character, key signature or just specify \code{"sharp"} or
#' \code{"flat"}.
#' @param tuning character, instrument tuning.
#' @param to_notes logical, return only the mapped notes character string
#' rather than the entire phrase object.
#' @param bar logical, insert a bar check at the end of the phrase.
#' @param ... arguments passed to \code{sf_phrase}.
#'
#' @return a phrase.
#' @seealso \code{\link{phrase}}
#' @export
#'
#' @examples
#' sf_phrase("5 4 3 2 1", "1 3 3 3 1", "8*4 1", key = "b_")
#' sf_phrase("654321 6s 12 1 21", "133211 355333 11 (13) (13)(13)", "4 4 8 8 4",
#'           key = "f")
#' sfp("6s*2 1*4", "000232*2 2*4", "4 4 8*4", tuning = "dropD", key = "d")
#'
#' # compare with single-argument input
#' s <- "3s*5 53~*3 543*2 643"
#' f <- "987*2 775 553 335 77~*3 545 325 210"
#' i <- "2*3 4. 16 4.*3 4*3"
#' p1 <- sfp(s, f, i)
#'
#' # Nominally shorter syntax, but potentially much easier to reason about
#' p2 <- sfp("3;987;2*2 775 ;553;4. ;335;16 5;7x7;4.~*3 ;545;4 325 6;2x10;")
#'
#' identical(p1, p2)
sf_phrase <- function(string, fret = NULL, info = NULL, key = "c",
                      tuning = "standard", to_notes = FALSE, bar = FALSE){
  if((is.null(fret) & !is.null(info)) | (!is.null(fret) & is.null(info)))
    stop("`fret` and `info` must both be provided or both be NULL.",
         call. = FALSE)
  if(is.null(fret)){
    sfi <- .split_sfp_input(string)
    string <- paste(sfi$string, collapse = " ")
    fret <- paste(sfi$fret, collapse = " ")
    info <- paste(sfi$info, collapse = " ")
  }
  .check_phrase_input(string, "string")
  .check_phrase_input(fret, "fret")
  string <- paste(gsub("_", "", .strsub(string)), collapse = " ")
  fret <- (strsplit(fret, " ")[[1]] %>%
             purrr::map_chr(.star_expand) %>%
              paste0(collapse = " ") %>%
             strsplit(" "))[[1]]
  tuning <- .map_tuning(tuning)
  open_notes <- rev(strsplit(tuning, " ")[[1]])
  str_num <- rev(seq_along(open_notes))
  notes <- purrr::map2(strsplit(string, " ")[[1]], fret, ~({
    string_tie <- grepl("~", .x)
    fret_tie <- grepl("~", .y)
    if(!identical(string_tie, fret_tie))
      stop("Tied note mismatch.", call. = FALSE)
    x <- if(any(string_tie)) gsub("~", "", .x) else .x
    y <- if(any(fret_tie)) gsub("~", "", .y) else .y
    rests <- c("r", "s")
    if(x %in% rests | y %in% rests){
      if(x == y) return(x) else stop("Rest mismatch.", call. = FALSE)
    }
    x <- as.integer(strsplit(x, "")[[1]])
    if(any(!x %in% str_num))
      stop("String number outside range inferred by tuning.", call. = FALSE)
    y <- trimws(gsub("(\\(\\d{1,2}\\))", " \\1 ", y))
    y <- strsplit(y, " ")[[1]]
    y <- lapply(y, function(x){
      if(substr(x, 1, 1) == "("){
        gsub("\\(|\\)", "", x)
      } else {
        strsplit(x, "")[[1]]
      }
      }) %>%
      unlist %>%
      as.integer
    if(length(x) != length(y)) stop("String/fret mismatch.", call. = FALSE)
    x <- sapply(seq_along(x), function(i, x, y){
      transpose(open_notes[x[i]], y[i], key, "tick")
    }, x = x, y = y)
    if(any(string_tie)) x[string_tie] <- paste0(x[string_tie], "~")
    paste(x, collapse = "")
  })) %>%
    unlist %>%
    paste(collapse = " ")
  if(to_notes) return(notes)
  phrase(notes, info, gsub("~", "", string), bar)
}

#' @export
#' @rdname sf_phrase
sfp <- sf_phrase

#' @export
#' @rdname sf_phrase
sf_note <- function(...){
  o <- list(...)
  o$to_notes <- TRUE
  do.call(sf_phrase, o)
}

#' @export
#' @rdname sf_phrase
sfn <- sf_note

.split_sfp_input <- function(x){
  x <- strsplit(x, " ")[[1]]
  mult <- sapply(x, .get_mult, USE.NAMES = FALSE)
  x <- sapply(x, .strip_mult, USE.NAMES = FALSE)
  if(any(!sapply(gregexpr(";", x), function(x) length(x[x != -1])) %in%
         c(0, 2)))
    stop("Must have 2 or 0 `;` present for a timestep to avoid ambiguity.",
         call. = FALSE)
  no_sep <- !grepl(";", x)
  if(any(no_sep)) x[no_sep] <- paste0(";", x[no_sep], ";")
  x <- gsub(";$", "; ", x)
  x <- lapply(strsplit(x, ";"), function(x) ifelse(x %in% c("", " "),
                                                   NA_character_, x))
  if(any(is.na(sapply(x, "[", 2))))
    stop("Fret values cannot be missing from `string;fret;info`", call. = FALSE)
  if(is.na(x[[1]][1]) | is.na(x[[1]][3]))
    stop("First timestep must include all three values as `string;fret;info`.",
         call. = FALSE)
  x <- do.call(rbind, x) %>%
    as.data.frame(stringsAsFactors = FALSE) %>%
    stats::setNames(c("string", "fret", "info")) %>%
    dplyr::as_tibble() %>%
    tidyr::fill(.data[["string"]], .data[["info"]]) %>%
    dplyr::mutate(
      string = .infer_strings(.data[["string"]], .data[["fret"]]),
      fret = gsub("x", "", .data[["fret"]])) %>%
    dplyr::slice(rep(seq_len(length(x)), times = mult))
  idx <- grep("~", x$info)
  if(length(idx)){
    x$string[idx] <- paste0(x$string[idx], "~")
    x$fret[idx] <- paste0(x$fret[idx], "~")
    x$info[idx] <- gsub("~", "", x$info[idx])
  }
  x
}

.n_frets <- function(x) nchar(gsub("(\\(\\d{1,2}\\))", ".", x))

.muted_index <- function(x) gregexpr("x", gsub("(\\(\\d{1,2}\\))", ".", x))

.infer_strings <- function(string, fret){
  n <- .n_frets(fret)
  muted <- .muted_index(fret)
  f <- function(string, fret, n, muted){
    if(n == 1 & fret %in% c("r", "s")) return(fret)
    if(string == n & -1 %in% muted & n > 2) return(paste0(string, "s"))
    string <- seq(as.integer(string), by = -1, length.out = n)
    if(!(-1 %in% muted)) string <- string[-muted]
    paste0(string, collapse = "")
  }
  purrr::map_chr(seq_along(string), ~{
    f(string[.x], fret[.x], n[.x], muted[[.x]])
  })
}

.get_mult <- function(x){
  y <- !grepl("\\*", x)
  if(any(y)) x[y] <- paste0(x[y], "*1")
  as.integer(gsub(".*\\*(\\d+)$", "\\1", x))
}

.strip_mult <- function(x) gsub("(.*)\\*\\d+$", "\\1", x)
