% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/id.cvm.R
\name{id.cvm}
\alias{id.cvm}
\title{Independence-based identification of SVAR models based on Cramer-von Mises distance}
\usage{
id.cvm(x, dd = NULL, itermax = 500, steptol = 100, iter2 = 75)
}
\arguments{
\item{x}{An object of class 'vars', 'vec2var', 'nlVar'. Estimated VAR object}

\item{dd}{Object of class 'indepTestDist' (generated by 'indepTest' from package 'copula'). A simulated independent sample of the same size as the data. If not supplied, it will be calculated by the function}

\item{itermax}{Integer. IMaximum number of iterations for DEoptim}

\item{steptol}{Numeric. Tolerance for steps without improvement for DEoptim}

\item{iter2}{Integer. Number of iterations for the second optimization}
}
\value{
A list of class "svars" with elements
\item{B}{Estimated structural impact matrix B, i.e. unique decomposition of the covariance matrix of reduced form errors}
\item{A_hat}{Estimated VAR parameter}
\item{method}{Method applied for identification}
\item{n}{Number of observations}
\item{type}{Type of the VAR model, e.g. 'const'}
\item{y}{Data matrix}
\item{p}{Number of lags}
\item{K}{Dimension of the VAR}
\item{rotation_angles}{Rotation angles, which lead to maximum independence}
\item{inc}{Indicator. 1 = second optimization increased the estimation precision. 0 = second optimization did not increase the estimation precision}
\item{test.stats}{Computed test statistics of independence test}
\item{iter1}{Number of iterations of first optimization}
\item{test1}{Minimum test statistic from first optimization}
\item{test2}{Minimum test statistic from second optimization}
}
\description{
Given an estimated VAR model, this function applies independence-based identification for the structural impact matrix B of the corresponding SVAR model
\deqn{y_t=c_t+A_1 y_{t-1}+...+A_p y_{t-p}+u_t   =c_t+A_1 y_{t-1}+...+A_p y_{t-p}+B \epsilon_t.}
Matrix B corresponds to the unique decomposition of the least squares covariance matrix \eqn{\Sigma_u=B B'} if the vector of structural shocks \eqn{\epsilon_t} contains at most one Gaussian shock (Comon, 1994).
A nonparametric dependence measure, the Cramer-von Mises distance (Genest and Remillard, 2004), determines least dependent structural shocks. The minimum is obtained by a two step optimization algorithm similar to the technique described in Herwartz and Ploedt (2016).
}
\examples{
\donttest{
# data contains quarterly observations from 1965Q1 to 2008Q3
# x = output gap
# pi = inflation
# i = interest rates
set.seed(23211)
v1 <- vars::VAR(USA, lag.max = 10, ic = "AIC" )
cob <- copula::indepTestSim(v1$obs, v1$K, verbose=FALSE)
x1 <- id.cvm(v1, dd = cob)
summary(x1)

# switching columns according to sign pattern
x1$B <- x1$B[,c(3,2,1)]
x1$B[,3] <- x1$B[,3]*(-1)

# impulse response analysis
i1 <- irf(x1, n.ahead = 30)
plot(i1, scales = 'free_y')
}

}
\references{
Herwartz, H., 2018. Hodges Lehmann detection of structural shocks - An Analysis of macroeconomic dynamics in the Euro Area, Oxford Bulletin of Economics and Statistics\cr
 Herwartz, H. & Ploedt, M., 2016. The macroeconomic effects of oil price shocks: Evidence from a statistical identification approach, Journal of International Money and Finance, 61, 30-44\cr
 Comon, P., 1994. Independent component analysis, A new concept?, Signal Processing, 36, 287-314\cr
 Genest, C. & Remillard, B., 2004. Tests of independence and randomness based on the empirical copula process, Test, 13, 335-370\cr
}
\seealso{
For alternative identification approaches see \code{\link{id.st}}, \code{\link{id.garch}}, \code{\link{id.cv}}, \code{\link{id.dc}} or \code{\link{id.ngml}}
}
