#' Code sidekickParse
#' 
#' Translate the tree representation generated by 
#' the R parser (parse) into a rectangular representation
#' @export
#' @param file File to parse
#' @param encoding the character encoding to use
#' @return A data frame with the following columns : 
#' - id : Identifier of the current node
#' - srcref1, ..., srcref4 : identifies the position of the current node in the file
#' - description : textual representation of the current node
#' - parent : identifier of the parent node
#' @author Romain Francois \email{francoisromain@@free.fr}
#' @examples
#' tf <- tempfile()
#' cat( "jitter <- " , deparse( jitter ), sep = "\n", file = tf )
#' sidekick( tf )
#' unlink( tf )
#' TODO :make this S3 instead of dispatching
sidekick <- function( file, encoding = getOption("encoding") ){
	
	if( file %of% "function" ){
		tf <- tempfile( ); on.exit( unlink( tf ) )
		dump( "file" , tf )
		file <- tf
	}
	
	if( is.character(file) ){
		if( file %~% '^rwd:' ){
			file <- sub( 'rwd:', getwd(), file ) 
		}
	}
	
	### try to parse and return an error if failed
	p <- try( parse( file, encoding = encoding ), silent = TRUE )
	if( p %of% "try-error" ){
		return( list( type = "error", data = parseError( p ) ) )
	}
	
	### calls the actual sidekick function
	out <- sidekickParse( p )
	# TODO: replace this in the code so that no conversion is necessary at that point

	out$id <- as.integer( out$id )
	out$srcref1 <- as.integer( out$srcref1 )
	out$srcref2 <- as.integer( out$srcref2 )
	out$srcref3 <- as.integer( out$srcref3 )
	out$srcref4 <- as.integer( out$srcref4 )
	out$parent  <- as.integer( out$parent )
	list( type = "ok", data = out )
}


sidekickParse <- function( p = try( parse(file), silent = TRUE) , top = TRUE, env = new.env(), parent = 0, file ){
	if( top ) {		
		env[["data"]] <- data.frame( 
			id = integer(0), 
			srcref1 = integer(0), 
			srcref2 = integer(0),
			srcref3 = integer(0),
			srcref4 = integer(0),
			description = character(0), 
			parent = integer(0), 
			mode = character(0), stringsAsFactors = FALSE )
		if( p %of% "try-error" ){
			return( env[["data"]] )
		}		
		maxId <- 0
	} else {
		maxId <- max( env[["data"]][, "id"] ) 
	}
	
	isIf <- looksLikeAnIf( p )
	atts <- attributes( p )
	descriptions <- as.character( p )
	hasAttrs <- "srcref" %in% names(atts)
	ids <- maxId + 1:length(p)
	if( hasAttrs ){
	  srcrefs <- t( sapply( attr(p, "srcref"), function( y ) {
			as.integer(y)[1:4]
			# if( length(positions) == 4 ) positions else positions[ c(1,5,3,6) ]
	  } ) )  
		colnames( srcrefs ) <- paste("srcref", 1:4 , sep = "")
		srcrefs <- as.data.frame( srcrefs ) 
		modes <- sapply( p, mode )
		data <- data.frame( id = ids, 
			parent = rep( parent, length(p) ), 
			mode = modes, srcrefs, description = descriptions, 
			stringsAsFactors = FALSE)
		env[["data"]] <- rbind( env[["data"]], data )
	} 
	
	calls <- sapply( p, mode ) %in% c("call","function")
  
	if( isIf ) {
		env[["data"]][ parent, "mode" ] <- "if"
		pa <- try( addIfNode( TRUE, env = env, parent = parent, p[[3]] ), silent = TRUE )
		sidekickParse( p[[3]], top = FALSE, env = env, parent = pa )  
		if( length(p) == 4) {
			pa <- try( addIfNode( FALSE, env = env, parent = parent, p[[4]] ), silent = TRUE )
			if( looksLikeAnIf( p[[4]]) ){
				data <- data.frame( 
					id          = pa + 1,
					getIfSrcref( p[[4]] ),
					description = paste( "if(", as.character( p[[4]][[2]]) , ")", sep = "" ), 
					parent      = pa , 
					mode        =  "if" )
				env[["data"]] <- rbind( env[["data"]], data )
				env[["data"]][ pa, 2:5] <- data[, 2:5] 
				pa <- pa + 1
			}
			sidekickParse( p[[4]], top = FALSE, env = env, parent = pa )  
		}
	} else{
		for( i in 1:length(p)){
			if( !is.null(p) && calls[i] ){
				test <- try( looksLikeAFunction( p[[i]] ), silent = TRUE )
				if( test ){
					env[["data"]][ ids[i], "mode" ] <- "function"
					try( sidekickParse( p[[i]], top = FALSE, env = env, parent = if( hasAttrs) ids[i] else parent ), silent = TRUE )  
				} else {
					sidekickParse( p[[i]], top = FALSE, env = env, parent = if( hasAttrs) ids[i] else parent )
				}
			}
		}
	}
	if( top ){
		env[["data"]]
	}
	
}

getIfSrcref <- function( p ){
	x <- attr(p, "srcref" )
	if( is.null( x ) ){
		x <- attr( p[[3]], "srcref" )
		if( length(p) == 4 ){
			x <- append( x, attr(p[[4]], "srcref" ) )
		}
	}
	y <- lapply( x, as.integer)
	srcref <- c( head( y , 1 )[[1]][1:2], tail(y,1)[[1]][3:4] )
	data.frame( srcref1= srcref[1], srcref2 = srcref[2], srcref3 = srcref[3], 
	  srcref4 = srcref[4], stringsAsFactors = FALSE )
}

addIfNode <- function( value = T, env = env, parent, nextnode ){
	
	data <- env[["data"]]
	srcref <- attr(nextnode, "srcref")
	if( is.null( srcref ) ){
		if( !looksLikeAnIf( nextnode ) ){
			return(parent)
		} else{          
			srcref <- attr( nextnode[[3]], "srcref" )
			if( length(nextnode) == 4 ){
				srcref <- append( srcref, attr( nextnode[[3]], "srcref" ) )
			}
		}
	}
	id <- max(data$id) + 1
	lap.out <- lapply( srcref, as.integer )
	srcref <- t(c(  
	  head( lap.out ,1)[[1]][1:2], 
		tail( lap.out ,1)[[1]][3:4] ) ) 
	colnames( srcref ) <- paste( "srcref", 1:4, sep = "") 
	mode <- paste( "if", value, sep = ":" )
	description <- mode
	env[["data"]] <- rbind( env[["data"]], data.frame( id = id, srcref, description = description, mode = mode, parent = parent ) )
	id
	 
}

looksLikeAFunction <- function( p ) {
	if( length( p[[1]]) != 1 ) return(FALSE)
	if( ! as.character( p[[1]] ) %in% c("<-", "<<-", "=" ) ) return(FALSE)
	if( length( p ) <= 2 ) return(FALSE) 
	if( is.null( p[[3]] ) ) return(FALSE)
	if( length( p[[3]] ) == 1 ) return(FALSE)
	asc <- as.character( p[[3]][[1]] )
	if( length( asc ) > 1 || asc != "function" ) return(FALSE)
	TRUE
}

looksLikeAnIf <- function(p){
	if( length( p[[1]] ) != 1 ) return(FALSE)
	as.character(p[[1]]) == "if"
}


dump. <- function( data, id = 0, level = 0 ){
	offset <- paste( rep( "\t", level ), collapse = "" ) 
	ids <- data$id[ data$parent == id ]
	if( length( ids ) ){
		for( i in 1:length(ids) ){
			 description <- data$description[ids[i]]
			 if( description != "{" ) cat( offset, description, "\n" )
			 dump.( data, id = ids[i], level = level + 1)
		}
	}
}


