\name{epidataCS_aggregate}
\alias{epidataCS2sts}
\alias{as.epidata.epidataCS}

\title{Conversion (aggregation) of \code{"epidataCS"} to \code{"epidata"} or \code{"sts"}}

\description{
  Continuous-time continuous-space epidemic data stored in an object of
  class \code{"\link{epidataCS}"} can be aggregated in space or in space
  and time yielding an object of class \code{"\link{epidata}"} or
  \code{"\linkS4class{sts}"} for use of \code{\link{twinSIR}} or
  \code{\link{hhh4}} modelling, respectively.
}

\usage{
## aggregation in space and time over 'stgrid' for use of 'hhh4' models
epidataCS2sts(object, freq, start, neighbourhood,
              tiles = NULL, popcol.stgrid = NULL, popdensity = TRUE)

## aggregation in space for use of 'twinSIR' models
\method{as.epidata}{epidataCS}(data, tileCentroids, eps = 0.001, ...)
}

\arguments{
  \item{object, data}{an object of class \code{"\link{epidataCS}"}.}
  \item{freq,start}{see the description of the \code{"\linkS4class{sts}"} class.}
  \item{neighbourhood}{
    binary adjacency or neighbourhood-order matrix of the regions
    (\code{tiles}). If missing but \code{tiles} is given, a binary
    adjacency matrix will be auto-generated from \code{tiles} using
    functionality of the \pkg{spdep} package (see
    \code{\link{poly2adjmat}}).
    Since the \code{"neighbourhood"} slot in \code{"\linkS4class{sts}"}
    is actually optional, \code{neighbourhood=NULL} also works.
  }  
  \item{tiles}{
    object inheriting from \code{"\linkS4class{SpatialPolygons}"}
    representing the regions in \code{object$stgrid} (column
    \code{"tile"}). It will become the \code{"map"} slot of the
    resulting \code{"sts"} object.
    Its \code{row.names} must match \code{levels(object$stgrid$tile)}.
    If \code{neighbourhood} is provided, \code{tiles} is optional (not
    required for \code{hhh4}, but for plots of the resulting
    \code{"sts"} object). 
  }
  \item{popcol.stgrid}{
    single character or numeric value indexing the
    column in \code{object$stgrid} which contains the population data
    (counts or densities, depending on the \code{popdensity} argument).
    This will become the \code{"populationFrac"} slot (optional).}
  \item{popdensity}{
    logical indicating if the column referenced by
    \code{popcol.stgrid} contains population densities or absolute counts.
  }

  \item{tileCentroids}{
    a coordinate matrix of the region centroids (i.e., the result of
    \code{coordinates(tiles)}). Its row names must match
    \code{levels(data$stgrid$tile)}.
    This will be the coordinates used for the \dQuote{population} (i.e.,
    the \code{tiles} from \code{"\link{epidataCS}"}) in the
    discrete-space \code{\link{twinSIR}} modelling.
  }
  \item{eps}{
    numeric scalar for breaking tied removal and infection times between different
    individuals (tiles), which might occur during conversion from
    \code{"epidataCS"} to \code{"epidata"}. Rather dumb, this is simply done
    by subtracting \code{eps} from each tied removal time.
    One should consider other ways of breaking the tied event times.
  }
  \item{\dots}{unused (argument of the generic).}
}

\details{
  Some comments on the conversion from \code{"epidataCS"} to \code{"epidata"}:
  the conversion results into SIS epidemics only,
  i.e. the at-risk indicator is set to 1 immediately after
  recovery. A tile is considered infective if at least one individual
  within the tile is infective, otherwise it is susceptible.
  The lengths of the infectious periods are taken from
  \code{data$events$eps.t}. There will be no \code{f} columns in the resulting
  \code{"epidata"}. These must be generated by a subsequent call to
  \code{\link{as.epidata}} with desired \code{f}.
}

\value{
  \code{epidataCS2sts}: an object of class \code{"\linkS4class{sts}"}
  representing the multivariate time-series of the number of
  cases aggregated over \code{stgrid}.

  \code{as.epidata.epidataCS}: an object of class
  \code{"\link{epidata}"} representing an SIS epidemic in form of a
  multivariate point process (one for each region/\code{tile}).
}

\author{
Sebastian Meyer
}

\seealso{
  \code{\link{epidata}} and \code{\link{twinSIR}}

  \code{linkS4class{sts}} and \code{\link{hhh4}}.
}

\examples{
data("imdepi")
load(system.file("shapes", "districtsD.RData", package="surveillance"))

## convert imdepi point pattern into multivariate time series
imdepi_sts <- epidataCS2sts(imdepi, freq=12, start=c(2002,1),
                            neighbourhood=NULL, # not needed here
                            tiles=districtsD)
\dontshow{
stopifnot(isTRUE(all.equal(colSums(imdepi_sts@observed),
                           c(table(imdepi$events$tile)))))
}

## compare plots of monthly number of cases
opar <- par(mfrow=c(2,1))
suppressWarnings(plot(imdepi, "time", breaks=c(0,unique(imdepi$stgrid$stop))))
plot(imdepi_sts, type=observed~time, legend.opts=NULL)
par(opar)

## plot number of cases by district
plot(imdepi_sts, type=observed~1|unit, labels=FALSE)


## also test conversion to an SIS event history ("epidata") of the "tiles"
if (requireNamespace("intervals")) {
    imdepi_short <- subset(imdepi, time < 50)
    imdepi_short$stgrid <- subset(imdepi_short$stgrid, start < 50)
    imdepi_epidata <- as.epidata(imdepi_short,
                                 tileCentroids=coordinates(districtsD))
    summary(imdepi_epidata)
}
}

\keyword{spatial}
\keyword{manip}
\keyword{methods}
