  ## Oracle GBH:
  # This function returns p-values adjusted according to the Oracle Group
  # Benjamini-Hochberg procedure. For a detailed description of the procedure
  # and its properties, see the paper  "False Discovery Rate Control with
  # Groups" by Zhou, Zhao, and Hu. Notice that instead of using alpha^weighted,
  # as in the paper, I include the alpha-weighting term in the adjusted p-values,
  # so that I can compare each of the weighted p-values to the original alpha in
  # the final analysis.
  #
  # Input: 1) unadjp - [vector of numerics between 0 and 1] - The unadjusted p-values
  # that have been computed by testing
  # multiple hypotheses tests. This is the same as what one would input into the
  # multtest function mt.rawp2adjp(), for example.
  # 2) groups -[same as other groups vectors] A vector that contains the group labeling for each hypothesis where
  # the index of each hypothesis is the same as the index used for the unadjusted
  # p-values above. For example, if the first 3 hypotheses were in group 1 and
  # the last 2 were in group 2, then we would input (1, 1, 1, 2, 2) as the ``groups''
  # vector.
  # 3) pi.groups - [numerics between 0 and 1, length = # groups] - This is the vector of proportion of true null hypotheses that corresponds
  # to each group. A user can input an estimate of pi0 from another package here.
  # In the adaptive procedure, these proportions are estimated from
  # the data. For example, if the hypotheses were divided into two groups, and
  # we found that the proportion of true null hypotheses in the first group was 0.9
  # and the proportion of true null hypotheses in the second group was 0.6, then we
  # would input pi.groups = c(0.9, 0.6). This is the aspect of the Group-Benjamini
  # Hochberg procedure that gives it more information to operate on than the
  # standard BH procedure: We are able to direct attention to those hypotheses
  # that are in groups where the proportion of true null hypotheses is estimated
  # to be low.
  #
  # Output: 1) GBH.frame - A data frame containing information from each
  # step of the Group Benjamini-Hochberg procedure. We are able to see
  # the sorted raw p-values and the hypothesis index from the p-values vector
  # that each raw p-value corresponds to in the first two columns. In
  # the second two columns, we see the sorted weighted p-values generated by
  # the procedure along with the indices of the hypotheses that those
  # weighted p-values correspond to. In the final column, we see the
  # finalized Group Benjamini-Hochberg adjusted p-values. The hypotheses
  # that each of these adjusted p-values corresponds to is the same
  # as the weighted p-values hypothesis index (we can do this because
  # the adjustment from weighted p-values to GBH p-values does not
  # change the ordering of the p-values.)

  Oracle.GBH.adjp <- function(unadj.p, groups, pi.groups){

    # This is only used for final comparison in the resulting data frame

    unadj.sorting.p <- sort(unadj.p, index.return = TRUE)
    p.sorted <- unadj.sorting.p$x
    p.index <- unadj.sorting.p$ix

    p.weighted <- unadj.p
    N <- length(unadj.p)

    nGroups <- length(unique(groups))

    # I would like to compute the size of each of the groups

    n_g <- vector(length = nGroups)
    for(i in 1:nGroups){
      n_g[i] <- length(which(groups == i))
    }

    # This allows me to calculate pi_0, the proportion of nulls overall

    pi_0 <- 1/N*(sum(n_g*pi.groups))

    # The first part of the procedure involves weighting p-values. This is where the
    # known group structure information is being explicitly accounted for.

    for(i in 1:N){
      pi_g.i <- pi.groups[groups[i]]
      if(pi_g.i != 1){
        p.weighted[i] <- unadj.p[i]*(pi_g.i/(1-pi_g.i))
      } else {
        p.weighted[i] <- Inf # We never reject hypotheses when the estimated prop.
                               # of null hypotheses in that group is 1.
      }
    }

    # The second part of the procedure is exactly like Benjamini-Hochberg in that it
    # is a step-up procedure where we compared ordered p-values to some constant factor
    # times alpha, where the constant is determined by the position of the p-value in
    # the ordered list.

    sorting.weighted.p <- sort(p.weighted, index.return = TRUE)
    p.weighted <- sorting.weighted.p$x
    p.weighted.index <- sorting.weighted.p$ix

    temp <- p.weighted
    for(i in (N-1):1){
      temp[i] <- min(temp[i+1], min(N*(1-pi_0)/i*p.weighted[i], 1))
    }
    if(temp[N] > 1) temp[N] <- 1

    adjp <- temp
    if(length(names(p.sorted)) == 0){
      names(p.sorted) <- p.index
    }
    if(length(names(p.weighted)) == 0){
      names(p.weighted) <- p.weighted.index
    }

    GBH.frame <- list()
    GBH.frame[[1]] <- data.frame(unadj.index = names(p.sorted),
                                 rawp = p.sorted)
    rownames(GBH.frame[[1]]) <- NULL
    GBH.frame[[2]] <- data.frame(adjusted.p.index = names(p.weighted),
                                 GBH = adjp)
    rownames(GBH.frame[[2]]) <- NULL
    names(GBH.frame) <- c("unadjusted.p.values", "GBH.adjusted.p.values")

    return(GBH.frame)

  }
