% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/opt.R
\name{dopt}
\alias{dopt}
\title{Optimum Sample Allocation in Stratified Sampling Schemes}
\usage{
dopt(n, a, m = NULL, M = NULL, M_method = "rna")
}
\arguments{
\item{n}{(\code{number})\cr total sample size. A strictly positive scalar.}

\item{a}{(\code{numeric})\cr parameters \eqn{a_1, ..., a_H} of variance function
\eqn{D}. Strictly positive numbers.}

\item{m}{(\code{numeric} or \code{NULL})\cr lower bounds constraints optionally imposed
on strata sample sizes. If different than \code{NULL}, it is then required that
\code{n >= sum(m)}. Must be assigned with \code{NULL} (default) if \code{M} is not \code{NULL}.
Strictly positive numbers.}

\item{M}{(\code{numeric} or \code{NULL})\cr upper bounds constraints optionally imposed
on strata sample sizes. If different than \code{NULL}, it is then required that
\code{n <= sum(M)}. Must be assigned with \code{NULL} (default) if \code{m} is not \code{NULL}.
Strictly positive numbers.}

\item{M_method}{(\code{string})\cr the name of the underlying algorithm to be used
for computing a sample allocation under one-sided upper bounds constraints
(Case I). One of the following: \emph{"rna"} (default), \emph{"sga"},
\emph{"sgaplus"}, \emph{"coma"}.}
}
\value{
Numeric vector with optimal sample allocation in strata.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

A classical problem in survey methodology in stratified sampling
is an optimum sample allocation problem. This problem is formulated as
determination of a vector of strata sample sizes that minimizes the
variance of the pi-estimator of the population total of a given study
variable, under constraint on total sample size.

The \code{dopt()} function solves the problem of optimum sample allocation
under either lower or upper bounds constraints, optionally imposed on
strata sample sizes. The allocation computed is valid for all stratified
sampling schemes for which the variance of the stratified pi-estimator is
of the form
\deqn{D(x_1,...,x_H) = a^2_1/x_1 + ... + a^2_H/x_H - b,}
where \eqn{H} denotes total number of strata, \eqn{x_1, ..., x_H} are the
strata sample sizes, and \eqn{b}, \eqn{a_w > 0} do not depend on
\eqn{x_w, w = 1, ..., H}. \cr

The \code{dopt()} function makes use of the following allocation algorithms:
\code{rNa}, \code{sga}, \code{sgaplus}, \code{coma} for optimal sample allocation under upper
bounds constraints only, and \code{LrNa} for optimal sample allocation under
lower bounds constraints only. The \code{rNa}, \code{sga}, and \code{coma} are described
in Wesołowski et al. (2021), the \code{sgaplus} in Wójciak (2019), and the
\code{LrNa} is introduced in Wójciak (2022).
If no inequality constraints are added, then no any special
algorithm is used as the allocation is given as a closed form expression,
known as Neyman allocation
\deqn{x_w = a_w * n / (a_1 + ... + a_H), w = 1, ..., H.}\cr
}
\details{
The \code{dopt()} function computes
\deqn{argmin D(x_1,...,x_H),}
under the equality constraint imposed on total sample size
\deqn{x_1 + ... + x_H = n.}
Optionally, one of the following set of one-sided inequality constraints
can be added
\deqn{x_w <= M_w, w = 1,...,H, (Case I)}
or
\deqn{x_w >= m_w, w = 1,...,H, (Case II)}
where \eqn{n > 0} denotes overall sample size, \eqn{m_w > 0}, and
\eqn{M_w > 0, w = 1, ..., H}, are the lower and upper bounds respectively,
imposed on sample strata sizes.\cr
User of \code{dopt()} can choose whether the inequality constraints will be
added to the optimization problem or not. This is achieved with the proper
use of \code{m} and \code{M} arguments of the function. In case of no inequality
constraints to be added, \code{m} and \code{M} must be both specified as \code{NULL}
(default). If only upper bounds constraints should be added (Case I), user
specifies these bounds with \code{M} argument, while leaving \code{m} as \code{NULL}.
Finally, if only lower bounds constraints should be added (Case II), user
specifies these bounds with \code{m} argument, while leaving \code{M} as \code{NULL}.
At the moment, \code{dopt()} function does not allow to add box-constraints
simultaneously.

For the Case I, there are four different underlying algorithms available
to use. These are abbreviated as: "rNa" (\code{\link[=rna_one_sided]{rna_one_sided()}}),
"sga" (\code{\link[=sga]{sga()}}), "sgaplus" (\code{\link[=sgaplus]{sgaplus()}}), "coma" (\code{\link[=coma]{coma()}}). Functions
names that perform given algorithms are given in the brackets. See its help
page for more details. For the Case II, the "rNa" (\code{\link[=rna_one_sided]{rna_one_sided()}}) is
used.
}
\note{
For simple random sampling without replacement design in each stratum,
parameters of the variance function \eqn{D} are
\eqn{b = N_1 * S_1^2 + ... + N_H * S_H^2}, and \eqn{a_w = N_w * S_w}, where
\eqn{N_w, S_w, w = 1, ..., H}, are strata sizes and standard deviations of
a study variable in strata respectively.
}
\examples{
a <- c(3000, 4000, 5000, 2000)
bounds <- c(100, 90, 70, 80)

# Only lower bounds.
dopt(n = 340, a = a, m = bounds)
dopt(n = 400, a = a, m = bounds)
dopt(n = 600, a = a, m = bounds)

# Only upper bounds.
dopt(n = 190, a = a, M = bounds)
dopt(n = 300, a = a, M = bounds)
dopt(n = 340, a = a, M = bounds)

# Example of execution-time comparison of different algorithms
# using bench R package.
\dontrun{
N <- pop969[, "N"]
S <- pop969[, "S"]
a <- N * S
nfrac <- seq(0.01, 0.9, 0.05)
n <- setNames(as.integer(nfrac * sum(N)), nfrac)
lapply(
  n,
  function(ni) {
    bench::mark(
      dopt(ni, a, M = N, M_method = "rna"),
      dopt(ni, a, M = N, M_method = "sga"),
      dopt(ni, a, M = N, M_method = "sgaplus"),
      dopt(ni, a, M = N, M_method = "coma"),
      iterations = 200
    )[c(1, 3)]
  }
)
}
}
\references{
Wesołowski, J., Wieczorkowski, R., Wójciak, W. (2021),
Optimality of the recursive Neyman allocation,
\emph{Journal of Survey Statistics and Methodology},
\doi{10.1093/jssam/smab018},
\doi{10.48550/arXiv.2105.14486} \cr

Wójciak, W. (2022),
Minimum sample size allocation in stratified sampling under constraints on
variance and strata sample sizes,
\doi{10.48550/arXiv.2204.04035} \cr

Wójciak, W. (2019), Optimal allocation in stratified sampling schemes,
\emph{MSc Thesis}, Warsaw University of Technology, Warsaw, Poland.
\url{http://home.elka.pw.edu.pl/~wwojciak/msc_optimal_allocation.pdf} \cr

Sarndal, C.-E., Swensson, B., and Wretman, J. (1992),
\emph{Model Assisted Survey Sampling}, New York, NY: Springer.
}
\seealso{
\code{\link[=nopt]{nopt()}}, \code{\link[=rna_one_sided]{rna_one_sided()}}, \code{\link[=sga]{sga()}}, \code{\link[=sgaplus]{sgaplus()}}, \code{\link[=coma]{coma()}}.
}
