#'Simulations using \code{rfca()}
#'
#'\code{rfca_options} repeatedly executes the \code{rfca()} function
#'over a number of different argument values and returns its solutions  
#'
#'@usage rfca_options(data, rfca.options.list, verbose)
#'
#'@param data a data frame
#'@param rfca.options.list a named list containing \code{rfca()} arguments
#'and the values with which they should be simulated; list should be initially
#'produced by a call to \code{rfca_args()}
#'@param verbose logical; if TRUE, will print extra execution informaton
#'
#'@return Returns a named list containing the following elements:
#'
#'\item{\code{opts}}{a data frame containing the parameter values used for each execution}
#'\item{\code{results}}{a list of \code{rfca()} solutions}
#'
#'
#'@examples
#'
#'# Load data
#'data(hicks_20)
#'
#'# Generate list of eqmcc arguments
#'arglist<-rfca_args()
#'
#'# Specify values for execution
#'arglist$outcome<-"CON"
#'arglist$case.ids<-"Case"
#'arglist$ntree<-c(1,5,100,5000)
#'arglist$mtry<-c(1,5,10,20)
#'
#'# Run execution
#'a<-rfca_options(data = hicks_20, rfca.options.list = arglist)
#'
#'# Get data frame of parameter values
#'a$opts
#'
#'# Get the solutions for parameter specifications in row 6
#'a$opts[4,]
#'a$results[[4]]
#'
rfca_options<-function(data, rfca.options.list, verbose){

  ### Alter function argument values, if necessary
  if(missing(data)){
    stop("The 'data' argument has not been specified", call. = F)
  }
  if(missing(rfca.options.list)){
    stop("The 'rfca.options.list' argument has not been specified", call. = F)
  }
  if(missing(verbose)){
    verbose<-FALSE
  }
  
  # Create options grid
  opts.grid<-expand.grid(rfca.options.list, stringsAsFactors = F)
  opts.grid<-opts.grid[,apply(X = opts.grid, MARGIN = 2, FUN = function(x){any(!is.na(x))})]
  opts.grid$list.id<-1:nrow(opts.grid)
  
  out<-lapply(X = 1:nrow(opts.grid), FUN = function(x){
    if(verbose){
      print(opts.grid[x,])
    }
    # Create list of options to feed to rfca
    opts<-as.list(opts.grid[x,])
    opts<-lapply(opts, function(x){x[[1]]})
    
    # Add dataset to the options list
    opts$data<-data
    
    # Do call to eqmcc
    sol<-tryCatch(expr = {
      do.call("rfca", opts)
    }, error=function(e) NULL)
    if(!is.null(sol)){
      sol
    }else{
      "No Solution"
    }
  })
  
  return(list(opts=opts.grid,
              results=out))
}