#' Version of mutate that (i) transforms data.table in place (ii) allows by, i condition 
#' 
#' @param x a data.table
#' @param ... Variables to include. Defaults to all non-grouping variables. 
#' @param i a condition that restricts the set of row on which to apply the function
#' @param by Groups within which the function should be applied
#' @param vars Used to work around non-standard evaluation.
#' @examples
#' library(data.table)
#' library(dplyr)
#' N <- 100
#' DT <- data.table(
#'   id = sample(5, N, TRUE),
#'   v1 = sample(5, N, TRUE),
#'   v2 = sample(1e6, N, TRUE)
#' )
#' setmutate(DT, mean(v2), by = v1)
#' setmutate_(DT, setNames("mean(v2)", "v3"), by = "v1")
#' setmutate_(DT, setNames("mean(v2)", "v3"), i = "id==1", by = "v1")
#' @export
setmutate <- function(x, ..., i = NULL, by = NULL){
    setmutate_(x, vars = lazy_dots(...), i = substitute(i), by = substitute(by))
}

#' @export
#' @rdname setmutate
setmutate_ <- function(x, vars, i = NULL, by = NULL){
    stopifnot(is.data.table(x))
    byvars <- names(select_vars_(names(x), by))
    if (!length(by)){
        byvars <- NULL
    }
    if (!is.null(i)){
      i <- as.lazy(i)$expr
    }
    dots <- lazyeval::all_dots(vars, all_named = TRUE)
    env <- dt_env(x, lazyeval::common_env(dots), byvars)
     # For each new variable, generate a call of the form df[, new := expr]
     for(col in names(dots)) {
      if (is.null(byvars) & is.null(i)){
         call <- substitute(dt[, lhs := rhs],
           list(lhs = as.name(col), rhs = dots[[col]]$expr))
       } else if (is.null(byvars) & !is.null(i)){
        call <- substitute(dt[i, lhs := rhs],
          list(lhs = as.name(col), rhs = dots[[col]]$expr, i = i))
      } else if (!is.null(byvars) & is.null(i)){
        call <- substitute(dt[, lhs := rhs, by = c(byvars)],
          list(lhs = as.name(col), rhs = dots[[col]]$expr))
      } else {
        call <- substitute(dt[i, lhs := rhs, by = c(byvars)],
          list(lhs = as.name(col), rhs = dots[[col]]$expr, i = i))
      }
    eval(call, env)
    }
  x[]
}

#' Version of mutate_each that (i) transforms data.table in place (ii) allows by, i condition (iii) when only when fun, creates new variables - except when replace = TRUE
#' 
#' @param x a data.table
#' @param funs List of function calls, generated by funs, or a character vector of function names.
#' @param ... Variables to include. Defaults to all non-grouping variables. 
#' @param i a condition that restricts the set of row on which to apply the function
#' @param by Groups within which the function should be applied
#' @param replace If replace = TRUE, variables are replaced. Default to FALSE (new variable are created)
#' @param vars Used to work around non-standard evaluation.
#' @examples
#' library(data.table)
#' library(dplyr)
#' N <- 100
#' DT <- data.table(
#'   id = sample(5, N, TRUE),
#'   v1 = sample(5, N, TRUE),
#'   v2 = sample(1e6, N, TRUE)
#' )
#' setmutate_each(DT, funs(mean, mean_na = mean(., na.rm = TRUE)), v2, i = id ==1, by = v1)
#' setmutate_each(DT, funs(as.character), replace = TRUE)
#' @export
setmutate_each <- function(x, funs, ..., i = NULL, by = NULL, replace = FALSE){
    setmutate_each_(x, funs, vars = lazy_dots(...), i = substitute(i), by = substitute(by), replace = replace)
}

#' @export
#' @rdname setmutate_each
setmutate_each_ <- function(x, funs, vars, i = NULL, by = NULL, replace = FALSE){
  stopifnot(is.data.table(x))
    if (anyDuplicated(names(funs))){
      stop("Multiple functions are specified with the same name", call. = FALSE)
    }
    if (replace & length(funs)!=1){
      stop("replace is TRUE but not one function is specified", call. = FALSE)
    }
    byvars <- names(select_vars_(names(x), by))
    if (!length(by)){
        byvars <- NULL
    }
    vars <- lazyeval::all_dots(vars)
    vars <- colwise_(x, funs_(funs), vars, byvars = byvars, replace = replace)
    setmutate_(x, vars, i, by)
}



colwise_ <- function(tbl, calls, vars, byvars = NULL, replace = FALSE) {
  vars <- select_vars_(tbl_vars(tbl), vars, exclude = byvars)
  if (!length(vars)){
    vars <- setdiff(names(tbl), byvars)
  }
  out <- vector("list", length(vars) * length(calls))
  dim(out) <- c(length(vars), length(calls))
  for (i in seq_along(vars)) {
    for (j in seq_along(calls)) {
      out[[i, j]] <- lazyeval::interp(calls[[j]],
        .values = list(. = as.name(vars[i])))
    }
  }
  dim(out) <- NULL  
  # modification is here:
  if (length(calls) == 1 & replace) {
    names(out) <- names(vars)
  } else {
    grid <- expand.grid(var = names(vars), call = names(calls))
    names(out) <- paste(grid$var, grid$call, sep = "_")
  }
  out
}


