% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TriTraining.R
\name{triTraining}
\alias{triTraining}
\title{Train the Tri-training model}
\usage{
triTraining(x, y, bclassif = bClassifOneNN(), dist = "matrix")
}
\arguments{
\item{x}{A object that can be coerced as matrix.
This object have various interpretations depending on the value set in \code{dist} argument.
See \code{dist} argument.}

\item{y}{A vector with the labels of training instances. In this vector the unlabeled instances
are specified with the value \code{NA}.}

\item{bclassif}{Base classifier specification. Default is \code{bClassifOneNN()}. For defining new base classifiers
see \code{\link{bClassif}}.}

\item{dist}{Distance information. Valid options are:
\itemize{
   \item \code{"matrix"}: this string indicates that \code{x} is a distance matrix.
   \item \emph{string}: the name of a distance method available in \code{proxy} package.
   In this case \code{x} is interpreted as a matrix of instances.
   \item \emph{function}: a function defined by the user that computes the distance between two vectors.
   This function is called passing the vectors in the firsts two arguments. If the function have others arguments,
   those arguments must be have default values.
   In this case \code{x} is interpreted as a matrix of instances.
}}
}
\value{
The trained model stored in a list with the following named values:
\describe{
  \item{models}{A list with the models (classifiers) trained.}
  \item{bclassif}{The value of \code{bclassif} argument.}
  \item{dist}{The value of \code{dist} argument.
  This value is optionally returned when the argument \code{dist} is different from "matrix".}
  \item{included.insts}{The indexes of the training instances used.
  This set is the union of the indexes used to train each model in \code{models}.
  These indexes are relative to \code{x} argument.}
  \item{indexes}{ A list of vectors. Each vector indicates the indexes of the instances
  used to train the corresponding model in \code{models}. For example, the instances used to train
  \code{models[1]} are \code{x[included.insts[indexes[1]], ]}.}
  \item{tr.insts}{The training instances corresponding to the indexes returned in \code{included.insts}.
   This value is optionally returned when the \code{dist} argument is different from "matrix".}
  \item{labels.map}{An internal map used for the labels.}
}
}
\description{
Builds and trains a model to predict the label of instances,
according to Tri-training algorithm.
}
\examples{
# This example is part of TriTraining demo.
# Use demo(TriTraining) to see all the examples.

## Load Wine data set
data(wine)

x <- wine[, -14] # instances without classes
y <- wine[, 14] # the classes
x <- scale(x) # scale the attributes

## Prepare data
set.seed(20)
# Use 50\% of instances for training
tra.idx <- sample(x = length(y), size = ceiling(length(y) * 0.5))
xtrain <- x[tra.idx,] # training instances
ytrain <- y[tra.idx]  # classes of training instances
# Use 70\% of train instances as unlabeled set
tra.na.idx <- sample(x = length(tra.idx), size = ceiling(length(tra.idx) * 0.7))
ytrain[tra.na.idx] <- NA # remove class information of unlabeled instances

# Use the other 50\% of instances for inductive testing
tst.idx <- setdiff(1:length(y), tra.idx)
xitest <- x[tst.idx,] # testing instances
yitest <- y[tst.idx] # classes of testing instances

## Example: Using the Euclidean distance in proxy package.
m <- triTraining(xtrain, ytrain, dist = "Euclidean")
pred <- predict(m, xitest)
caret::confusionMatrix(table(pred, yitest))

}
\references{
ZhiHua Zhou and Ming Li.\cr
\emph{Tri-training: exploiting unlabeled data using three classifiers.}\cr
IEEE Transactions on Knowledge and Data Engineering, 17(11):1529–1541, Nov 2005. ISSN 1041-4347. doi: 10.1109/TKDE.2005. 186.
}

