% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rolloff.R
\name{getRolloff}
\alias{getRolloff}
\title{Control rolloff of harmonics}
\usage{
getRolloff(
  pitch_per_gc = c(440),
  nHarmonics = NULL,
  rolloff = -6,
  rolloffOct = 0,
  rolloffParab = 0,
  rolloffParabHarm = 3,
  rolloffParabCeiling = NULL,
  rolloffKHz = 0,
  baseline = 200,
  dynamicRange = 80,
  samplingRate = 16000,
  plot = FALSE
)
}
\arguments{
\item{pitch_per_gc}{a vector of f0 per glottal cycle, Hz}

\item{nHarmonics}{maximum number of harmonics to generate (very weak
harmonics with amplitude < \code{-dynamicRange} will be discarded)}

\item{rolloff}{basic rolloff from lower to upper harmonics, db/octave
(exponential decay). All rolloff parameters are in anchor format. See
\code{\link{getRolloff}} for more details}

\item{rolloffOct}{basic rolloff changes from lower to upper harmonics
(regardless of f0) by \code{rolloffOct} dB/oct. For example, we can get
steeper rolloff in the upper part of the spectrum}

\item{rolloffParab}{an optional quadratic term affecting only the first
\code{rolloffParabHarm} harmonics. The middle harmonic of the first
\code{rolloffParabHarm} harmonics is amplified or dampened by
\code{rolloffParab} dB relative to the basic exponential decay}

\item{rolloffParabHarm}{the number of harmonics affected by
\code{rolloffParab}}

\item{rolloffParabCeiling}{quadratic adjustment is applied only up to
\code{rolloffParabCeiling}, Hz. If not NULL, it overrides
\code{rolloffParabHarm}}

\item{rolloffKHz}{rolloff changes linearly with f0 by \code{rolloffKHz}
dB/kHz. For ex., -6 dB/kHz gives a 6 dB steeper basic rolloff as f0 goes up
by 1000 Hz}

\item{baseline}{The "neutral" f0, at which no adjustment of rolloff
takes place regardless of \code{rolloffKHz}}

\item{dynamicRange}{dynamic range, dB. Harmonics and noise more than
dynamicRange under maximum amplitude are discarded to save computational
resources}

\item{samplingRate}{sampling rate (needed to stop at Nyquist frequency and
for plotting purposes)}

\item{plot}{if TRUE, produces a plot}
}
\value{
Returns a matrix of amplitude multiplication factors for adjusting
  the amplitude of harmonics relative to f0 (1 = no adjustment, 0 = silent).
  Each row of output contains one harmonic, and each column contains one
  glottal cycle.
}
\description{
Harmonics are generated as separate sine waves. But we don't want each
harmonic to be equally strong, so we normally specify some rolloff function
that describes the loss of energy in upper harmonics relative to the
fundamental frequency (f0). \code{\link{getRolloff}} provides flexible
control over this rolloff function, going beyond simple exponential decay
(\code{rolloff}). Use quadratic terms to modify the behavior of a few lower
harmonics, \code{rolloffOct} to adjust the rate of decay per
octave, and \code{rolloffKHz} for rolloff correction depending on
f0. Plot the output with different parameter values and see examples below
and the vignette to get a feel for how to use \code{\link{getRolloff}}
effectively.
}
\examples{
# steady exponential rolloff of -12 dB per octave
rolloff = getRolloff(pitch_per_gc = 150, rolloff = -12,
  rolloffOct = 0, rolloffKHz = 0, plot = TRUE)
# the rate of rolloff slows down by 1 dB each octave
rolloff = getRolloff(pitch_per_gc = 150, rolloff = -12,
  rolloffOct = 1, rolloffKHz = 0, plot = TRUE)

# rolloff can be made to depend on f0 using rolloffKHz
rolloff = getRolloff(pitch_per_gc = c(150, 400, 800),
  rolloffOct = 0, rolloffKHz = -3, plot = TRUE)
# without the correction for f0 (rolloffKHz),
  # high-pitched sounds have the same rolloff as low-pitched sounds,
  # producing unnaturally strong high-frequency harmonics
rolloff = getRolloff(pitch_per_gc = c(150, 400, 800),
  rolloffOct = 0, rolloffKHz = 0, plot = TRUE)

# parabolic adjustment of lower harmonics
rolloff = getRolloff(pitch_per_gc = 350, rolloffParab = 0,
  rolloffParabHarm = 2, plot = TRUE)
# rolloffParabHarm = 1 affects only f0
rolloff = getRolloff(pitch_per_gc = 150, rolloffParab = 30,
  rolloffParabHarm = 1, plot = TRUE)
# rolloffParabHarm=2 or 3 affects only h1
rolloff = getRolloff(pitch_per_gc = 150, rolloffParab = 30,
  rolloffParabHarm = 2, plot = TRUE)
# rolloffParabHarm = 4 affects h1 and h2, etc
rolloff = getRolloff(pitch_per_gc = 150, rolloffParab = 30,
  rolloffParabHarm = 4, plot = TRUE)
# negative rolloffParab weakens lower harmonics
rolloff = getRolloff(pitch_per_gc = 150, rolloffParab = -20,
  rolloffParabHarm = 7, plot = TRUE)
# only harmonics below 2000 Hz are affected
rolloff = getRolloff(pitch_per_gc = c(150, 600),
  rolloffParab = -20, rolloffParabCeiling = 2000,
  plot = TRUE)

# dynamic rolloff (varies over time)
rolloff = getRolloff(pitch_per_gc = c(150, 250),
                     rolloff = c(-12, -18, -24), plot = TRUE)
rolloff = getRolloff(pitch_per_gc = c(150, 250), rolloffParab = 40,
                    rolloffParabHarm = 1:5, plot = TRUE)

\dontrun{
# Note: getRolloff() is called internally by soundgen()
# using the data.frame format for all vectorized parameters
# Compare:
s1 = soundgen(sylLen = 1000, pitch = 250,
              rolloff = c(-24, -2, -18), plot = TRUE)
s2 = soundgen(sylLen = 1000, pitch = 250,
              rolloff = data.frame(time = c(0, .2, 1),
                                   value = c(-24, -2, -18)),
              plot = TRUE)

# Also works for rolloffOct, rolloffParab, etc:
s3 = soundgen(sylLen = 1000, pitch = 250,
             rolloffParab = 20, rolloffParabHarm = 1:15, plot = TRUE)
}
}
\seealso{
\code{\link{soundgen}}
}
