% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sjTabItemAnalysis.R
\name{sjt.itemanalysis}
\alias{sjt.itemanalysis}
\title{Summary of item analysis of an item scale as HTML table}
\usage{
sjt.itemanalysis(df, factor.groups = NULL,
  factor.groups.titles = "auto", scale = FALSE,
  min.valid.rowmean = 2, alternate.rows = TRUE, sort.column = NULL,
  show.shapiro = FALSE, show.kurtosis = FALSE,
  show.corr.matrix = TRUE, CSS = NULL, encoding = NULL,
  file = NULL, use.viewer = TRUE, remove.spaces = TRUE)
}
\arguments{
\item{df}{A data frame with items.}

\item{factor.groups}{If not \code{NULL}, \code{df} will be splitted into sub-groups,
where the item analysis is carried out for each of these groups. Must be a vector of same
length as \code{ncol(df)}, where each item in this vector represents the group number of
the related columns of \code{df}. If \code{factor.groups = "auto"}, a principal
component analysis with Varimax rotation is performed, and the resulting
groups for the components are used as group index. See 'Examples'.}

\item{factor.groups.titles}{Titles for each factor group that will be used as table caption for each
component-table. Must be a character vector of same length as \code{length(unique(factor.groups))}.
Default is \code{"auto"}, which means that each table has a standard caption \emph{Component x}.
Use \code{NULL} to suppress table captions.}

\item{scale}{Logical, if \code{TRUE}, the data frame's vectors will be scaled when calculating the
Cronbach's Alpha value (see \code{\link[performance]{item_reliability}}). Recommended, when
the variables have different measures / scales.}

\item{min.valid.rowmean}{Minimum amount of valid values to compute row means for index scores.
Default is 2, i.e. the return values \code{index.scores} and \code{df.index.scores} are
computed for those items that have at least \code{min.valid.rowmean} per case (observation, or
technically, row). See \code{mean_n} for details.}

\item{alternate.rows}{Logical, if \code{TRUE}, rows are printed in
alternatig colors (white and light grey by default).}

\item{sort.column}{Numeric vector, indicating the index of the column
that should sorted. by default, the column is sorted in ascending order.
Use negative index for descending order, for instance,
\code{sort.column = -3} would sort the third column in descending order.
Note that the first column with rownames is not counted.}

\item{show.shapiro}{Logical, if \code{TRUE}, a Shapiro-Wilk normality test is computed for each item.
See \code{\link{shapiro.test}} for details.}

\item{show.kurtosis}{Logical, if \code{TRUE}, the kurtosis for each item will also be shown (see \code{\link[psych]{kurtosi}}
and \code{\link[psych]{describe}} in the \code{psych}-package for more details.}

\item{show.corr.matrix}{Logical, if \code{TRUE} (default), a correlation matrix of each component's
index score is shown. Only applies if \code{factor.groups} is not \code{NULL} and \code{df} has
more than one group. First, for each case (df's row), the sum of all variables (df's columns) is
scaled (using the \code{\link{scale}}-function) and represents a "total score" for
each component (a component is represented by each group of \code{factor.groups}).
After that, each case (df's row) has a scales sum score for each component.
Finally, a correlation of these "scale sum scores" is computed.}

\item{CSS}{A \code{\link{list}} with user-defined style-sheet-definitions,
according to the \href{http://www.w3.org/Style/CSS/}{official CSS syntax}.
See 'Details' or \href{../doc/table_css.html}{this package-vignette}.}

\item{encoding}{Character vector, indicating the charset encoding used
for variable and value labels. Default is \code{"UTF-8"}. For Windows
Systems, \code{encoding = "Windows-1252"} might be necessary for proper
display of special characters.}

\item{file}{Destination file, if the output should be saved as file.
If \code{NULL} (default), the output will be saved as temporary file and
openend either in the IDE's viewer pane or the default web browser.}

\item{use.viewer}{Logical, if \code{TRUE}, the HTML table is shown in the IDE's
viewer pane. If \code{FALSE} or no viewer available, the HTML table is
opened in a web browser.}

\item{remove.spaces}{Logical, if \code{TRUE}, leading spaces are removed from all lines in the final string
that contains the html-data. Use this, if you want to remove parantheses for html-tags. The html-source
may look less pretty, but it may help when exporting html-tables to office tools.}
}
\value{
Invisibly returns
        \itemize{
         \item \code{df.list}: List of data frames with the item analysis for each sub.group (or complete, if \code{factor.groups} was \code{NULL})
         \item \code{index.scores}: A data frame with of standardized scale / index scores for each case (mean value of all scale items for each case) for each sub-group.
         \item \code{ideal.item.diff}: List of vectors that indicate the ideal item difficulty for each item in each sub-group. Item difficulty only differs when items have different levels.
         \item \code{cronbach.values}: List of Cronbach's Alpha values for the overall item scale for each sub-group.
         \item \code{knitr.list}: List of html-tables with inline-css for use with knitr for each table (sub-group)
         \item \code{knitr}: html-table of all complete output with inline-css for use with knitr
         \item \code{complete.page}: Complete html-output.
         }
         If \code{factor.groups = NULL}, each list contains only one elment, since just one
         table is printed for the complete scale indicated by \code{df}. If \code{factor.groups}
         is a vector of group-index-values, the lists contain elements for each sub-group.
}
\description{
This function performs an item analysis with certain statistics that are
               useful for scale or index development. The resulting tables are shown in the
               viewer pane resp. webbrowser or can be saved as file. Following statistics are
               computed for each item of a data frame:
               \itemize{
                 \item percentage of missing values
                 \item mean value
                 \item standard deviation
                 \item skew
                 \item item difficulty
                 \item item discrimination
                 \item Cronbach's Alpha if item was removed from scale
                 \item mean (or average) inter-item-correlation
               }
               Optional, following statistics can be computed as well:
               \itemize{
                 \item kurstosis
                 \item Shapiro-Wilk Normality Test
               }
               If \code{factor.groups} is not \code{NULL}, the data frame \code{df} will be
               splitted into groups, assuming that \code{factor.groups} indicate those columns
               of the data frame that belong to a certain factor (see return value of function \code{\link{sjt.pca}}
               as example for retrieving factor groups for a scale and see examples for more details).
}
\note{
\itemize{
         \item The \emph{Shapiro-Wilk Normality Test} (see column \code{W(p)}) tests if an item has a distribution that is significantly different from normal.
         \item \emph{Item difficulty} should range between 0.2 and 0.8. Ideal value is \code{p+(1-p)/2} (which mostly is between 0.5 and 0.8).
         \item For \emph{item discrimination}, acceptable values are 0.20 or higher; the closer to 1.00 the better. See \code{\link[performance]{item_reliability}} for more details.
         \item In case the total \emph{Cronbach's Alpha} value is below the acceptable cut-off of 0.7 (mostly if an index has few items), the \emph{mean inter-item-correlation} is an alternative measure to indicate acceptability. Satisfactory range lies between 0.2 and 0.4. See also \code{\link[performance]{item_intercor}}.
       }
}
\examples{
# Data from the EUROFAMCARE sample dataset
library(sjmisc)
library(sjlabelled)
data(efc)

# retrieve variable and value labels
varlabs <- get_label(efc)

# recveive first item of COPE-index scale
start <- which(colnames(efc) == "c82cop1")
# recveive last item of COPE-index scale
end <- which(colnames(efc) == "c90cop9")

# create data frame with COPE-index scale
mydf <- data.frame(efc[, start:end])
colnames(mydf) <- varlabs[start:end]

\dontrun{
sjt.itemanalysis(mydf)

# auto-detection of labels
sjt.itemanalysis(efc[, start:end])

# Compute PCA on Cope-Index, and perform a
# item analysis for each extracted factor.
indices <- sjt.pca(mydf)$factor.index
sjt.itemanalysis(mydf, factor.groups = indices)

# or, equivalent
sjt.itemanalysis(mydf, factor.groups = "auto")}

}
\references{
\itemize{
             \item Jorion N, Self B, James K, Schroeder L, DiBello L, Pellegrino J (2013) Classical Test Theory Analysis of the Dynamics Concept Inventory. (\href{https://www.academia.edu/4104752/Classical_Test_Theory_Analysis_of_the_Dynamics_Concept_Inventory}{web})
             \item Briggs SR, Cheek JM (1986) The role of factor analysis in the development and evaluation of personality scales. Journal of Personality, 54(1), 106-148. \doi{10.1111/j.1467-6494.1986.tb00391.x}
             \item McLean S et al. (2013) Stigmatizing attitudes and beliefs about bulimia nervosa: Gender, age, education and income variability in a community sample. International Journal of Eating Disorders. \doi{10.1002/eat.22227}
             \item Trochim WMK (2008) Types of Reliability. (\href{http://www.socialresearchmethods.net/kb/reltypes.php}{web})
            }
}
