#  Copyright (c) 2023 Merck & Co., Inc., Rahway, NJ, USA and its affiliates.
#  All rights reserved.
#
#  This file is part of the simtrial program.
#
#  simtrial is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

#' Simulated survival dataset with delayed treatment effect
#'
#' Magirr and Burman (2019) considered several scenarios for their
#' modestly weighted logrank test.
#' One of these had a delayed treatment effect with a hazard ratio
#' of 1 for 6 months followed by a hazard ratio of 1/2 thereafter.
#' The scenario enrolled 200 patients uniformly over 12 months and
#' cut data for analysis 36 months after enrollment was opened.
#' This dataset was generated by the [sim_pw_surv()] function
#' under the above scenario.
#'
#' @format
#' A data frame with 200 rows and 4 columns:
#' - `tte`: Time to event.
#'
#' @references
#' Magirr, Dominic, and Carl‐Fredrik Burman. 2019.
#' "Modestly weighted logrank tests."
#' _Statistics in Medicine_ 38 (20): 3782--3790.
#'
#' @examples
#' library(survival)
#' library(dplyr)
#'
#' fit <- survfit(Surv(tte, event) ~ treatment, data = mb_delayed_effect)
#'
#' # Plot survival
#' plot(fit, lty = 1:2)
#' legend("topright", legend = c("control", "experimental"), lty = 1:2)
#'
#' # Set up time, event, number of event dataset for testing
#' # with arbitrary weights
#' ten <- mb_delayed_effect |> counting_process(arm = "experimental")
#' head(ten)
#'
#' # MaxCombo with logrank, FH(0,1), FH(1,1)
#' ten |>
#'   fh_weight(rho_gamma = data.frame(rho = c(0, 0, 1), gamma = c(0, 1, 1)), return_corr = TRUE) |>
#'   pvalue_maxcombo()
#'
#' # Magirr-Burman modestly down-weighted rank test with 6 month delay
#' # First, add weights
#' ten <- ten |> mb_weight(6)
#' head(ten)
#'
#' # Now compute test based on these weights
#' ten |>
#'   summarize(
#'     S = sum(o_minus_e * mb_weight),
#'     V = sum(var_o_minus_e * mb_weight^2),
#'     Z = S / sqrt(V)
#'   ) |>
#'   mutate(p = pnorm(Z))
#'
#' # Create 0 weights for first 6 months
#' ten <- ten |> mutate(w6 = 1 * (tte >= 6))
#' ten |>
#'   summarize(
#'     S = sum(o_minus_e * w6),
#'     V = sum(var_o_minus_e * w6^2),
#'     Z = S / sqrt(V)
#'   ) |>
#'   mutate(p = pnorm(Z))
#'
#' # Generate another dataset
#' ds <- sim_pw_surv(
#'   n = 200,
#'   enroll_rate = data.frame(rate = 200 / 12, duration = 12),
#'   fail_rate = data.frame(
#'     stratum = c("All", "All", "All"),
#'     period = c(1, 1, 2),
#'     treatment = c("control", "experimental", "experimental"),
#'     duration = c(42, 6, 36),
#'     rate = c(log(2) / 15, log(2) / 15, log(2) / 15 * 0.6)
#'   ),
#'   dropout_rate = data.frame(
#'     stratum = c("All", "All"),
#'     period = c(1, 1),
#'     treatment = c("control", "experimental"),
#'     duration = c(42, 42),
#'     rate = c(0, 0)
#'   )
#' )
#' # Cut data at 24 months after final enrollment
#' mb_delayed_effect_2 <- ds |> cut_data_by_date(max(ds$enroll_time) + 24)
"mb_delayed_effect"
