% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simStateSpace-sim-ssm-var-i-vary.R
\name{SimSSMVARIVary}
\alias{SimSSMVARIVary}
\title{Simulate Data from a Vector Autoregressive Model
using a State Space Model Parameterization
for n > 1 Individuals (Individual-Varying Parameters)}
\usage{
SimSSMVARIVary(
  n,
  mu0,
  sigma0,
  alpha,
  beta,
  psi,
  gamma_eta = NULL,
  x = NULL,
  time = 0,
  burn_in = 0
)
}
\arguments{
\item{n}{Positive integer.
Number of individuals.}

\item{mu0}{List of numeric vectors.
Each element of the list
is the mean of initial latent variable values
(\eqn{\boldsymbol{\mu}_{\boldsymbol{\eta} \mid 0}}).}

\item{sigma0}{List of numeric matrices.
Each element of the list
is the covariance matrix
of initial latent variable values
(\eqn{\boldsymbol{\Sigma}_{\boldsymbol{\eta} \mid 0}}).}

\item{alpha}{List of numeric vectors.
Each element of the list
is the vector of intercepts for the dynamic model
(\eqn{\boldsymbol{\alpha}}).}

\item{beta}{List of numeric matrices.
Each element of the list
is the transition matrix relating the values of the latent variables
at time \code{t - 1} to those at time \code{t}
(\eqn{\boldsymbol{\beta}}).}

\item{psi}{List of numeric matrices.
Each element of the list
is the process noise covariance matrix
(\eqn{\boldsymbol{\Psi}}).}

\item{gamma_eta}{Numeric matrix.
Matrix relating the values of the covariate matrix
at time \code{t} to the latent variables at time \code{t}
(\eqn{\boldsymbol{\Gamma}_{\boldsymbol{\eta}}}).}

\item{x}{A list of length \code{n} of numeric matrices.
Each element of the list
is a matrix of observed covariates in \code{type = 1} or \code{type = 2}.
The number of rows in each matrix should be equal to \code{time + burn_in}.}

\item{time}{Positive integer.
Number of time points to simulate.}

\item{burn_in}{Positive integer.
Number of burn-in points to exclude before returning the results.}
}
\value{
Returns an object of class \code{simstatespace}
which is a list with the following elements:
\itemize{
\item \code{call}: Function call.
\item \code{args}: Function arguments.
\item \code{data}: Generated data which is a list of length \code{n}.
Each element of \code{data} is a list with the following elements:
\itemize{
\item \code{id}: A vector of ID numbers of length \code{t}.
\item \code{time}: A vector time points of length \code{t}.
\item \code{y}: A \code{t} by \code{k} matrix of values for the manifest variables.
\item \code{eta}: A \code{t} by \code{p} matrix of values for the latent variables.
\item \code{x}: A \code{t} by \code{j} matrix of values for the covariates.
}
\item \code{fun}: Function used.
}
}
\description{
This function simulates data from a vector autoregressive model
using a state space model parameterization
for \code{n > 1} individuals.
In this model,
the parameters can vary across individuals.
}
\details{
Parameters can vary across individuals
by providing a list of parameter values.
If the length of any of the parameters
(\code{mu0},
\code{sigma0},
\code{alpha},
\code{beta},
\code{psi}, or
\code{gamma_eta})
is less the \code{n},
the function will cycle through the available values.
}
\examples{
# prepare parameters
# In this example, beta varies across individuals
set.seed(42)
k <- 3
iden <- diag(k)
null_vec <- rep(x = 0, times = k)
n <- 5
mu0 <- list(null_vec)
sigma0 <- list(iden)
alpha <- list(null_vec)
beta <- list(
  diag(x = 0.1, nrow = k),
  diag(x = 0.2, nrow = k),
  diag(x = 0.3, nrow = k),
  diag(x = 0.4, nrow = k),
  diag(x = 0.5, nrow = k)
)
psi <- list(iden)
time <- 50
burn_in <- 0
gamma_eta <- list(0.10 * diag(k))
x <- lapply(
  X = seq_len(n),
  FUN = function(i) {
    return(
      matrix(
        data = rnorm(n = k * (time + burn_in)),
        ncol = k
      )
    )
  }
)

# No covariates
ssm <- SimSSMVARIVary(
  n = n,
  mu0 = mu0,
  sigma0 = sigma0,
  alpha = alpha,
  beta = beta,
  psi = psi,
  time = time,
  burn_in = burn_in
)

plot(ssm)

# With covariates
ssm <- SimSSMVARIVary(
  n = n,
  mu0 = mu0,
  sigma0 = sigma0,
  alpha = alpha,
  beta = beta,
  psi = psi,
  gamma_eta = gamma_eta,
  x = x,
  time = time,
  burn_in = burn_in
)

plot(ssm)

}
\references{
Chow, S.-M., Ho, M. R., Hamaker, E. L., & Dolan, C. V. (2010).
Equivalence and differences between structural equation modeling
and state-space modeling techniques.
\emph{Structural Equation Modeling: A Multidisciplinary Journal},
17(2), 303–332.
\doi{10.1080/10705511003661553}
}
\seealso{
Other Simulation of State Space Models Data Functions: 
\code{\link{OU2SSM}()},
\code{\link{SimSSMFixed}()},
\code{\link{SimSSMIVary}()},
\code{\link{SimSSMLinGrowthIVary}()},
\code{\link{SimSSMLinGrowth}()},
\code{\link{SimSSMOUFixed}()},
\code{\link{SimSSMOUIVary}()},
\code{\link{SimSSMOU}()},
\code{\link{SimSSMVARFixed}()},
\code{\link{SimSSMVAR}()},
\code{\link{SimSSM}()}
}
\author{
Ivan Jacob Agaloos Pesigan
}
\concept{Simulation of State Space Models Data Functions}
\keyword{sim}
\keyword{simStateSpace}
\keyword{var}
