% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/morphers.R
\name{spatial_morphers}
\alias{spatial_morphers}
\alias{to_spatial_contracted}
\alias{to_spatial_directed}
\alias{to_spatial_explicit}
\alias{to_spatial_neighborhood}
\alias{to_spatial_shortest_paths}
\alias{to_spatial_simple}
\alias{to_spatial_smooth}
\alias{to_spatial_subdivision}
\alias{to_spatial_subset}
\alias{to_spatial_transformed}
\title{Spatial morphers for sfnetworks}
\usage{
to_spatial_contracted(
  x,
  ...,
  simplify = FALSE,
  summarise_attributes = "ignore",
  store_original_data = FALSE
)

to_spatial_directed(x)

to_spatial_explicit(x, ...)

to_spatial_neighborhood(x, node, threshold, weights = NULL, from = TRUE, ...)

to_spatial_shortest_paths(x, ...)

to_spatial_simple(
  x,
  remove_multiple = TRUE,
  remove_loops = TRUE,
  summarise_attributes = "first",
  store_original_data = FALSE
)

to_spatial_smooth(x, store_original_data = FALSE)

to_spatial_subdivision(x)

to_spatial_subset(x, ..., subset_by = NULL)

to_spatial_transformed(x, ...)
}
\arguments{
\item{x}{An object of class \code{\link{sfnetwork}}.}

\item{...}{Arguments to be passed on to other functions. See the description
of each morpher for details.}

\item{simplify}{Should the network be simplified after contraction? This
means that multiple edges and loop edges will be removed. Multiple edges
are introduced by contraction when there are several connections between
the same groups of nodes. Loop edges are introduced by contraction when
there are connections within a group. Note however that setting this to
\code{TRUE} also removes multiple edges and loop edges that already
existed before contraction. Defaults to \code{FALSE}.}

\item{summarise_attributes}{Whenever multiple features (i.e. nodes and/or
edges) are merged into a single feature during morphing, how should their
attributes be combined? Several options are possible, see
\code{\link[igraph]{igraph-attribute-combination}} for details.}

\item{store_original_data}{Whenever multiple features (i.e. nodes and/or
edges) are merged into a single feature during morphing, should the data of
the original features be stored as an attribute of the new feature, in a
column named \code{.orig_data}. This is in line with the design principles
of \code{tidygraph}. Defaults to \code{FALSE}.}

\item{node}{The geospatial point for which the neighborhood will be
calculated. Can be an integer, referring to the index of the node for which
the neighborhood will be calculated. Can also be an object of class
\code{\link[sf]{sf}} or \code{\link[sf]{sfc}}, containing a single feature.
In that case, this point will be snapped to its nearest node before
calculating the neighborhood. When multiple indices or features are given,
only the first one is taken.}

\item{threshold}{The threshold distance to be used. Only nodes within the
threshold distance from the reference node will be included in the
neighborhood. Should be a numeric value in the same units as the weight
values used for distance calculation.}

\item{weights}{The edge weights used to calculate distances on the network.
Can be a numeric vector giving edge weights, or a column name referring to
an attribute column in the edges table containing those weights. If set to
\code{NULL}, the values of a column named \code{weight} in the edges table
will be used automatically, as long as this column is present. If not, the
geographic edge lengths will be calculated internally and used as weights.}

\item{from}{Should distances be calculated from the reference node towards
the other nodes? Defaults to \code{TRUE}. If set to \code{FALSE}, distances
will be calculated from the other nodes towards the reference node instead.}

\item{remove_multiple}{Should multiple edges be merged into one. Defaults
to \code{TRUE}.}

\item{remove_loops}{Should loop edges be removed. Defaults to \code{TRUE}.}

\item{subset_by}{Whether to create subgraphs based on nodes or edges.}
}
\value{
Either a \code{morphed_sfnetwork}, which is a list of one or more
\code{\link{sfnetwork}} objects, or a \code{morphed_tbl_graph}, which is a
list of one or more \code{\link[tidygraph]{tbl_graph}} objects. See the
description of each morpher for details.
}
\description{
Spatial morphers form spatial add-ons to the set of
\code{\link[tidygraph]{morphers}} provided by \code{tidygraph}. These
functions are not meant to be called directly. They should either be passed
into \code{\link[tidygraph]{morph}} to create a temporary alternative
representation of the input network. Such an alternative representation is a
list of one or more network objects. Single elements of that list can be
extracted directly as a new network by passing the morpher to
\code{\link[tidygraph]{convert}} instead, to make the changes lasting rather
than temporary. Alternatively, if the morphed state contains multiple
elements, all of them can be extracted together inside a
\code{\link[tibble]{tbl_df}} by passing the morpher to
\code{\link[tidygraph]{crystallise}}.
}
\details{
It also possible to create your own morphers. See the documentation
of \code{\link[tidygraph]{morph}} for the requirements for custom morphers.
}
\section{Functions}{
\itemize{
\item \code{to_spatial_contracted}: Combine groups of nodes into a single node per
group. \code{...} is forwarded to \code{\link[dplyr]{group_by}} to
create the groups. The centroid of the group of nodes will be used as
geometry of the contracted node. If edge are spatially explicit, edge
geometries are updated accordingly such that the valid spatial network
structure is preserved. Returns a \code{morphed_sfnetwork} containing a
single element of class \code{\link{sfnetwork}}.

\item \code{to_spatial_directed}: Make a network directed in the direction given
by the linestring geometries of the edges. Differs from
\code{\link[tidygraph]{to_directed}}, which makes a network directed based
on the node indices given in the \code{from} and \code{to} columns. In
undirected networks these indices may not correspond with the endpoints of
the linestring geometries. Returns a \code{morphed_sfnetwork} containing a
single element of class \code{\link{sfnetwork}}. This morpher requires edges
to be spatially explicit. If not, use \code{\link[tidygraph]{to_directed}}.

\item \code{to_spatial_explicit}: Create linestring geometries between source
and target nodes of edges. If the edges data can be directly converted to
an object of class \code{\link[sf]{sf}} using \code{\link[sf]{st_as_sf}},
extra arguments can be provided as \code{...} and will be forwarded to
\code{\link[sf]{st_as_sf}} internally. Otherwise, straight lines will be
drawn between the source and target node of each edge. Returns a
\code{morphed_sfnetwork} containing a single element of class
\code{\link{sfnetwork}}.

\item \code{to_spatial_neighborhood}: Limit a network to the spatial neighborhood of
a specific node. \code{...} is forwarded to
\code{\link[tidygraph]{node_distance_from}} (if \code{from} is \code{TRUE})
or \code{\link[tidygraph]{node_distance_to}} (if \code{from} is
\code{FALSE}). Returns a \code{morphed_sfnetwork} containing a single
element of class \code{\link{sfnetwork}}.

\item \code{to_spatial_shortest_paths}: Limit a network to those nodes and edges that
are part of the shortest path between two nodes. \code{...} is evaluated in
the same manner as \code{\link{st_network_paths}} with
\code{type = 'shortest'}. Returns a \code{morphed_sfnetwork} that may
contain multiple elements of class \code{\link{sfnetwork}}, depending on
the number of requested paths. When unmorphing only the first instance of
both the node and edge data will be used, as the the same node and/or edge
can be present in multiple paths.

\item \code{to_spatial_simple}: Remove loop edges and/or merges multiple edges
into a single edge. Multiple edges are edges that have the same source and
target nodes (in directed networks) or edges that are incident to the same
nodes (in undirected networks). When merging them into a single edge, the
geometry of the first edge is preserved. The order of the edges can be
influenced by calling \code{\link[dplyr]{arrange}} before simplifying.
Returns a \code{morphed_sfnetwork} containing a single element of class
\code{\link{sfnetwork}}.

\item \code{to_spatial_smooth}: Construct a smoothed version of the network by
iteratively removing pseudo nodes, while preserving the connectivity of the
network. In the case of directed networks, pseudo nodes are those nodes that
have only one incoming and one outgoing edge. In undirected networks, pseudo
nodes are those nodes that have two incident edges. Connectivity of the
network is preserved by concatenating the incident edges of each removed
pseudo node. Returns a \code{morphed_sfnetwork} containing a single element
of class \code{\link{sfnetwork}}.

\item \code{to_spatial_subdivision}: Construct a subdivision of the network by
subdividing edges at each interior point that is equal to any
other interior or boundary point in the edges table. Interior points in this
sense are those points that are included in their linestring geometry
feature but are not endpoints of it, while boundary points are the endpoints
of the linestrings. The network is reconstructed after subdivision such that
edges are connected at the points of subdivision. Returns a
\code{morphed_sfnetwork} containing a single element of class
\code{\link{sfnetwork}}. This morpher requires edges to be spatially
explicit.

\item \code{to_spatial_subset}: Subset the network by applying a spatial
filter, i.e. a filter on the geometry column based on a spatial predicate.
\code{...} is evaluated in the same manner as \code{\link[sf]{st_filter}}.
Returns a \code{morphed_sfnetwork} containing a single element of class
\code{\link{sfnetwork}}. For filters on an attribute column, use
\code{\link[tidygraph]{to_subgraph}}.

\item \code{to_spatial_transformed}: Transform the geospatial coordinates of the
network into a different coordinate reference system. \code{...} is
evaluated in the same manner as \code{\link[sf]{st_transform}}.
Returns a \code{morphed_sfnetwork} containing a single element of class
\code{\link{sfnetwork}}.
}}

\examples{
library(sf, quietly = TRUE)
library(tidygraph, quietly = TRUE)

net = as_sfnetwork(roxel, directed = FALSE) \%>\%
  st_transform(3035)

# Temporary changes with morph and unmorph.
net \%>\%
 activate("edges") \%>\%
 mutate(weight = edge_length()) \%>\%
 morph(to_spatial_shortest_paths, from = 1, to = 10) \%>\%
 mutate(in_paths = TRUE) \%>\%
 unmorph()

# Lasting changes with convert.
net \%>\%
 activate("edges") \%>\%
 mutate(weight = edge_length()) \%>\%
 convert(to_spatial_shortest_paths, from = 1, to = 10)

}
\seealso{
The vignette on
\href{https://luukvdmeer.github.io/sfnetworks/articles/morphers.html}{spatial morphers}.
}
