% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clade_phylm.R
\name{clade_phylm}
\alias{clade_phylm}
\title{Influential clade detection - Phylogenetic Linear Regression}
\usage{
clade_phylm(formula, data, phy, model = "lambda", track = TRUE, clade.col,
  n.species = 5, times = 100, ...)
}
\arguments{
\item{formula}{The model formula}

\item{data}{Data frame containing species traits with row names matching tips
in \code{phy}.}

\item{phy}{A phylogeny (class 'phylo') matching \code{data}.}

\item{model}{The phylogenetic model to use (see Details). Default is \code{lambda}.}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{clade.col}{The name of a column in the provided data frame with clades 
specification (a character vector with clade names).}

\item{n.species}{Minimum number of species in the clade in order to include
this clade in the leave-one-out deletion analyis. Default is \code{5}.}

\item{times}{Number of simulations for the randomization test.}

\item{...}{Further arguments to be passed to \code{phylolm}}
}
\value{
The function \code{clade_phylm} returns a list with the following
components:

\code{formula}: The formula

\code{full.model.estimates}: Coefficients, aic and the optimised
value of the phylogenetic parameter (e.g. \code{lambda}) for the full model
without deleted species.

\code{clade.model.estimates}: A data frame with all simulation
estimates. Each row represents a deleted clade. Columns report the calculated
regression intercept (\code{intercept}), difference between simulation
intercept and full model intercept (\code{DFintercept}), the percentage of change
in intercept compared to the full model (\code{intercept.perc}) and intercept
p-value (\code{pval.intercept}). All these parameters are also reported for the regression
slope (\code{DFslope} etc.). Additionally, model aic value (\code{AIC}) and
the optimised value (\code{optpar}) of the phylogenetic parameter 
(e.g. \code{kappa} or \code{lambda}, depending on the phylogenetic model used) 
are reported.

\code{data}: Original full dataset.

\code{errors}: Clades where deletion resulted in errors.
}
\description{
Estimate the impact on model estimates of phylogenetic linear regression after 
removing clades from the analysis.
}
\details{
This function sequentially removes one clade at a time, fits a phylogenetic
linear regression model using \code{\link[phylolm]{phylolm}} and stores the
results. The impact of of a specific clade on model estimates is calculated by the
comparison between the full model (with all species) and the model without 
the species belonging to a clade.

Additionally, to account for the influence of the number of species on each 
clade (clade sample size), this function also estimate a null distribution of slopes
expected for the number of species in a given clade. This is done by fitting
 models without the same number of species in the given clade. 
 The number of simulations to be performed is set by 'times'. To test if the 
 clade influence differs from the null expectation, a randomization test can
 be performed using 'summary(x)'. 

All phylogenetic models from \code{phylolm} can be used, i.e. \code{BM},
\code{OUfixedRoot}, \code{OUrandomRoot}, \code{lambda}, \code{kappa},
\code{delta}, \code{EB} and \code{trend}. See ?\code{phylolm} for details.

\code{clade_phylm} detects influential clades based on
difference in intercept and/or slope when removing a given clade compared
to the full model including all species.

Currently, this function can only implement simple linear models (i.e. 
\eqn{y = a + bx}). In the future we will implement more complex models.

Output can be visualised using \code{sensi_plot}.
}
\examples{
\dontrun{
# Load data:
data(primates)
# run analysis:
clade <- clade_phylm(log(sexMaturity) ~ log(adultMass), 
phy = primates$phy[[1]], data = primates$data, times = 30, clade.col = "family")
# To check summary results and most influential clades:
summary(clade)
# Visual diagnostics for clade removal:
sensi_plot(clade)
# Specify which clade removal to plot:
sensi_plot(clade, "Cercopithecidae")
sensi_plot(clade, "Cebidae")
}
}
\author{
Gustavo Paterno
}
\references{
Ho, L. S. T. and Ane, C. 2014. "A linear-time algorithm for 
Gaussian and non-Gaussian trait evolution models". Systematic Biology 63(3):397-408.
}
\seealso{
\code{\link[phylolm]{phylolm}}, \code{\link[sensiPhy]{samp_phyglm}},
 \code{\link{influ_phylm}}, \code{\link{sensi_plot}}
\code{\link{sensi_plot}}
}

