#' seminr estimate_cbsem() function
#'
#' The \code{seminr} package provides a natural syntax for researchers to describe
#' structural equation models.
#'
#' @param data A \code{dataframe} containing the indicator measurement data.
#'
#' @param measurement_model A list represention of how constructs are measured
#'   by their items, generated using \code{\link{constructs}}.
#'   Note that only reflective constructs are supported for CBSEM models,
#'   though a composite measurement model can be converted into a reflective one
#'   using \code{\link{as.reflective}}.
#'
#' @param structural_model A source-to-target matrix representing the
#'   structural model, generated by \code{\link{relationships}}.
#'
#' @param item_associations An item-to-item matrix representing error
#'   covariances that are freed for estimation.
#'   This matrix is created by \code{associations()}, or defaults to NULL
#'   (no associations).
#'
#' @param estimator A character string indicating which estimation method to use
#'   in Lavaan. It defaults to "MLR" for robust estimation.
#'   See the Lavaan documentation for other supported estimators.
#'
#' @param ... Any other parameters to pass to \code{lavaan::sem} during
#'   estimation.
#'
#' @usage
#' estimate_cbsem(data, measurement_model, structural_model, item_associations=NULL,
#'                estimator="MLR", ...)
#'
#' @references Joreskog, K. G. (1973). A general method for estimating a linear structural equation system In: Goldberger AS, Duncan OD, editors. Structural Equation Models in the Social Sciences. New York: Seminar Press.
#'
#' @seealso \code{\link{as.reflective}}
#'          \code{\link{relationships}} \code{\link{constructs}}
#'          \code{\link{paths}}
#'          \code{\link{associations}} \code{\link{item_errors}}
#'
#' @examples
#' mobi <- mobi
#'
#' #seminr syntax for creating measurement model
#' mobi_mm <- constructs(
#'   reflective("Image",        multi_items("IMAG", 1:5)),
#'   reflective("Quality",      multi_items("PERQ", 1:7)),
#'   reflective("Value",        multi_items("PERV", 1:2)),
#'   reflective("Satisfaction", multi_items("CUSA", 1:3)),
#'   reflective("Complaints",   single_item("CUSCO")),
#'   reflective("Loyalty",      multi_items("CUSL", 1:3))
#' )
#'
#' #seminr syntax for freeing up item-item covariances
#' mobi_am <- associations(
#'   item_errors(c("PERQ1", "PERQ2"), "IMAG1")
#' )
#'
#' #seminr syntax for creating structural model
#' mobi_sm <- relationships(
#'   paths(from = c("Image", "Quality"), to = c("Value", "Satisfaction")),
#'   paths(from = c("Value", "Satisfaction"), to = c("Complaints", "Loyalty")),
#'   paths(from = "Complaints",   to = "Loyalty")
#' )
#'
#' # Estimate model and get results
#' mobi_cbsem <- estimate_cbsem(mobi, mobi_mm, mobi_sm, mobi_am)
#'
#' # Use or capture the summary object for more results and metrics
#' summary(mobi_cbsem)
#'
#' cbsem_summary <- summary(mobi_cbsem)
#' cbsem_summary$descriptives$correlations$constructs
#'
#' @export
estimate_cbsem <- function(data, measurement_model, structural_model, item_associations=NULL, estimator="MLR", ...) {
  cat("Generating the seminr model for CBSEM\n")

  # TODO: consider higher order models (see estimate_pls() function for template)

  post_interaction_object <- process_cbsem_interactions(measurement_model, data, structural_model, item_associations, estimator, ...)
  names(post_interaction_object$data) <- sapply(names(post_interaction_object$data), FUN=lavaanify_name, USE.NAMES = FALSE)
  mmMatrix <- post_interaction_object$mmMatrix
  data <- post_interaction_object$data

  # Rename interaction terms
  structural_model[, "source"] <- sapply(structural_model[, "source"], FUN=lavaanify_name)
  smMatrix <- structural_model

  # TODO: warning if the model is incorrectly specified
  # warnings(measurement_model, data, structural_model)

  # Create LAVAAN syntax
  measurement_syntax <- lavaan_mm_syntax(mmMatrix)
  structural_syntax <- lavaan_sm_syntax(smMatrix)
  association_syntax <- lavaan_item_associations(item_associations)

  # Put all the parts together
  full_syntax <- paste(measurement_syntax, structural_syntax, association_syntax, sep="\n\n")

  lavaan_model <- try_or_stop(
    lavaan::sem(
      model=full_syntax, data=data, std.lv = TRUE, estimator=estimator, ...),
    "run CBSEM in Lavaan"
  )

  # Inspect results
  constructs <- all_construct_names(measurement_model)
  lavaan_std <- lavaan::lavInspect(lavaan_model, what="std")
  loadings <- lavaan_std$lambda
  class(loadings) <- "matrix"
  tenB <- estimate_lavaan_ten_berge(lavaan_model)

  # Gather model information
  seminr_model <- list(
    data = data,
    measurement_model = measurement_model,
    mmMatrix = mmMatrix,
    smMatrix = smMatrix,
    factor_loadings = loadings,
    associations = item_associations,
    constructs = constructs,
    construct_scores = tenB$scores,
    item_weights = tenB$weights,
    lavaan_syntax = full_syntax,
    lavaan_model = lavaan_model
  )

  class(seminr_model) <- c("cbsem_model", "seminr_model")
  return(seminr_model)
}

#' seminr estimate_cfa() function
#'
#' Estimates a Confirmatory Factor Analysis (CFA) model
#'
#' @inheritParams estimate_cbsem
#'
#' @usage
#' estimate_cfa(data, measurement_model, item_associations=NULL, estimator="MLR", ...)
#'
#' @references Jöreskog, K.G. (1969) A general approach to confirmatory maximum likelihood factor analysis. Psychometrika, 34, 183-202.
#'
#' @seealso \code{\link{constructs}} \code{\link{reflective}}
#'          \code{\link{associations}} \code{\link{item_errors}}
#'          \code{\link{as.reflective}}
#'
#' #' @examples
#' mobi <- mobi
#'
#' #seminr syntax for creating measurement model
#' mobi_mm <- constructs(
#'   reflective("Image",        multi_items("IMAG", 1:5)),
#'   reflective("Expectation",  multi_items("CUEX", 1:3)),
#'   reflective("Quality",      multi_items("PERQ", 1:7))
#' )
#'
#' #seminr syntax for freeing up item-item covariances
#' mobi_am <- associations(
#'   item_errors(c("PERQ1", "PERQ2"), "CUEX3"),
#'   item_errors("IMAG1", "CUEX2")
#' )
#'
#' mobi_cfa <- estimate_cfa(mobi, mobi_mm, mobi_am)
#'
#' @export
estimate_cfa <- function(data, measurement_model, item_associations=NULL,
                         estimator="MLR", ...) {
  cat("Generating the seminr model for CFA\n")

  # TODO: consider higher order models (see estimate_pls() function for template)

  # TODO: warning if the model is incorrectly specified
  # warnings(measurement_model, data, structural_model)

  # Create LAVAAN syntax
  mmMatrix <- mm2matrix(measurement_model)

  measurement_syntax <- lavaan_mm_syntax(mmMatrix)
  association_syntax <- lavaan_item_associations(item_associations)

  full_syntax <- paste(measurement_syntax,
                       association_syntax,
                       sep="\n\n")

  # Run the model in LAVAAN
  lavaan_model <- try_or_stop(
    lavaan::cfa(model=full_syntax, data=data, std.lv = TRUE,
                estimator=estimator, ...),
    "run CFA in Lavaan"
  )

  tenB <- estimate_lavaan_ten_berge(lavaan_model)

  # Gather model information
  seminr_model <- list(
    data = data,
    measurement_model = measurement_model,
    lavaan_syntax = full_syntax,
    construct_scores = tenB$scores,
    item_weights = tenB$weights,
    lavaan_model = lavaan_model
  )

  class(seminr_model) <- c("cfa_model", "seminr_model")
  return(seminr_model)
}

