% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convenienceFunctions.R
\name{semPower.powerARMA}
\alias{semPower.powerARMA}
\title{semPower.powerARMA}
\usage{
semPower.powerARMA(
  type,
  comparison = "restricted",
  nWaves = NULL,
  autoregEffects = NULL,
  autoregLag1 = autoregEffects,
  autoregLag2 = NULL,
  autoregLag3 = NULL,
  mvAvgLag1 = NULL,
  mvAvgLag2 = NULL,
  mvAvgLag3 = NULL,
  means = NULL,
  variances = NULL,
  waveEqual = NULL,
  groupEqual = NULL,
  nullEffect = NULL,
  nullWhich = NULL,
  nullWhichGroups = NULL,
  invariance = TRUE,
  autocorResiduals = TRUE,
  ...
)
}
\arguments{
\item{type}{type of power analysis, one of \code{'a-priori'}, \code{'post-hoc'}, \code{'compromise'}.}

\item{comparison}{comparison model, one of \code{'saturated'} or \code{'restricted'} (the default). This determines the df for power analyses. \code{'saturated'} provides power to reject the model when compared to the saturated model, so the df equal the one of the hypothesized model. \code{'restricted'} provides power to reject the hypothesized model when compared to an otherwise identical model that just omits the restrictions defined in \code{nullEffect}, so the df equal the number of restrictions.}

\item{nWaves}{number of waves, must be >= 2.}

\item{autoregEffects}{vector of the lag-1 autoregressive effects, e.g. \code{c(.7, .6)} for  autoregressive effects of .7 for \code{X1 -> X2} and .6 for \code{X2 -> X3}. Must be a list for multiple groups models.}

\item{autoregLag1}{alternative name for autoregEffects.}

\item{autoregLag2}{vector of lag-2 effects, e.g. \code{c(.2, .1)} for lag-2 effects of .2 for \code{X1 -> X3} and .1 for \code{X2 -> X4}.}

\item{autoregLag3}{vector of lag-3 effects, e.g. \code{c(.2)} for a lag-3 effect of .2 for \code{X1 -> X4}.}

\item{mvAvgLag1}{vector of the lag-1 moving average parameters, e.g. \code{c(.4, .3)} for moving average parameters of .4 for \code{N1 -> X2} and .3 for \code{N2 -> X3}. Must be a list for multiple groups models.}

\item{mvAvgLag2}{vector of the lag-2 moving average parameters, e.g. \code{c(.3, .2)} for moving average parameters effects of .2 for \code{N1 -> X3} and .2 for \code{N2 -> X4}. Must be a list for multiple groups models.}

\item{mvAvgLag3}{vector of the lag-3  moving average parameters, e.g. \code{c(.2)} for a moving average parameter of .2 for \code{N1 -> X4}. Must be a list for multiple groups models.}

\item{means}{vector of means of \code{X}. May be \code{NULL} for no meanstructure.}

\item{variances}{vector of variances of the noise factors \code{N} (= residual variances of \code{X}).}

\item{waveEqual}{parameters that are assumed to be equal across waves in both the H0 and the H1 model. Because ARMA models are likely not identified when no such constraints are imposed, this may not be empty. Valid are \code{'autoreg'}, \code{'autoregLag2'}, and  \code{'autoregLag3'} for autoregressive effects, \code{'mvAvg'}, \code{'mvAvgLag2'}, and  \code{'mvAvgLag3'} for moving average effects, \code{var} for the variance of the noise factors (starting at wave 2), \code{mean} for the conditional means of X  (starting at wave 2).}

\item{groupEqual}{parameters that are restricted across groups in both the H0 and the H1 model, when \code{nullEffect} implies a multiple group model. Valid are \code{autoreg}, \code{mvAvg}, \code{var}, \code{mean}.}

\item{nullEffect}{defines the hypothesis of interest. Valid are the same arguments as in \code{waveEqual} and additionally \code{'autoreg = 0'}, \code{'autoregLag2 = 0'}, \code{'autoregLag3 = 0'}, \code{'mvAvg = 0'}, \code{'mvAvgLag2 = 0'}, \code{'mvAvgLag3 = 0'},  to constrain the autoregressive or moving average effects to zero, and \code{'autoregA = autoregB'}, \code{'mvAvgA = mvAvgB'}, \code{'varA = varB'}, \code{'meanA = meanB'} to constrain the autoregressive (lag-1) effects, moving average (lag-1) effects, variances of the noise factors, or means of the X to be equal across groups.}

\item{nullWhich}{used in conjunction with \code{nullEffect} to identify which parameter to constrain when there are multiple waves and parameters are not constant across waves. For example, \code{nullEffect = 'autoreg = 0'} with \code{nullWhich = 2} would constrain the second autoregressive effect for X to zero.}

\item{nullWhichGroups}{for hypothesis involving cross-groups comparisons, vector indicating the groups for which equality constrains should be applied, e.g. \code{c(1, 3)} to constrain the relevant parameters of the first and the third group. If \code{NULL}, all groups are constrained to equality.}

\item{invariance}{whether metric invariance over waves is assumed (\code{TRUE}, the default) or not (\code{FALSE}). When means are part of the model, invariant intercepts are also assumed. This affects the df when the comparison model is the saturated model and generally affects power (also for comparisons to the restricted model).}

\item{autocorResiduals}{whether the residuals of the indicators of latent variables are autocorrelated over waves (\code{TRUE}, the default) or not (\code{FALSE}). This affects the df when the comparison model is the saturated model and generally affects power (also for comparisons to the restricted model).}

\item{...}{mandatory further parameters related to the specific type of power analysis requested, see \code{\link[=semPower.aPriori]{semPower.aPriori()}}, \code{\link[=semPower.postHoc]{semPower.postHoc()}}, and \code{\link[=semPower.compromise]{semPower.compromise()}}, and parameters specifying the factor model. The order of factors is (X1, X2, ..., X_nWaves). See details.}
}
\value{
a list. Use the \code{summary} method to obtain formatted results. Beyond the results of the power analysis and a number of effect size measures, the list contains the following components:
\item{\code{Sigma}}{the population covariance matrix. A list for multiple group models.}
\item{\code{mu}}{the population mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{SigmaHat}}{the H0 model implied covariance matrix. A list for multiple group models.}
\item{\code{muHat}}{the H0 model implied mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{modelH0}}{\code{lavaan} H0 model string.}
\item{\code{modelH1}}{\code{lavaan} H1 model string or \code{NULL} when the comparison refers to the saturated model.}
\item{\code{simRes}}{detailed simulation results when a simulated power analysis (\code{simulatedPower = TRUE}) was performed.}
}
\description{
Convenience function for performing power analysis on effects in an ARMA model.
This requires the lavaan package.
}
\details{
This function performs a power analysis to reject various hypotheses arising
in models with autoregressive and moving average parameters (ARMA models), where one variable \code{X} is repeatedly
assessed at different time points (\code{nWaves}), and autoregressive (lag-1 effects; \code{X1 -> X2 -> X3}, and optionally lag-2 and lag-3) effects,
and moving average parameters (\code{N1 ->  X2}, or equivalently for lag-2 and lag-3 effects) are assumed.

Relevant hypotheses in arising in an ARMA model are:
\itemize{
\item \code{autoreg}: Tests the hypothesis that the autoregressive lag-1 effects are equal across waves (stationarity of autoregressive lag-1 effects).
\item \code{autoregLag2}: Tests the hypothesis that the autoregressive lag-2 effects are equal across waves (stationarity of autoregressive lag-2 effects).
\item \code{autoregLag3}: Tests the hypothesis that the autoregressive lag-3 effects are equal across waves (stationarity of autoregressive lag-3 effects).
\item \code{mvAvg}: Tests the hypothesis that the moving average lag-1 parameters are equal across waves (stationarity of moving average lag-1 effects).
\item \code{mvAvgLag2}: Tests the hypothesis that the moving average lag-2 parameters are equal across waves (stationarity of moving average lag-2 effects).
\item \code{mvAvgLag3}: Tests the hypothesis that the moving average lag-3 parameters are equal across waves (stationarity of moving average lag-3 effects).
\item \code{var}: Tests the hypothesis that the variances of the noise factors N (= the residual variances of X) are equal across waves 2 to nWaves (stationarity of variance).
\item \code{mean}: Tests the hypothesis that the conditional means of X are equal across waves 2 to nWaves (stationarity of means).
\item \code{autoreg = 0}, \code{autoregLag2 = 0}, \code{autoregLag3 = 0}: Tests the hypothesis that the autoregressive effects of the specified lag is zero.
\item \code{mvAvg = 0}, \code{mvAvgLag2 = 0}, \code{mvAvgLag3 = 0}: Tests the hypothesis that the moving average parameter of the specified lag is zero.
\item \code{autoregA = autoregB}: Tests the hypothesis that the autoregressive lag-1 effect is equal across groups.
\item \code{mvAvgA = mvAvgB}: Tests the hypothesis that the moving average lag-1 parameter is equal across groups.
\item \code{varA = varB}: Tests the hypothesis that the variance of the noise factors are equal across groups.
\item \code{meanA = meanB}: Tests the hypothesis that latent means are equal across groups.
}

For hypotheses regarding a simple autoregression, see \code{\link[=semPower.powerAutoreg]{semPower.powerAutoreg()}}. For hypotheses regarding a CLPM structure, see \code{\link[=semPower.powerCLPM]{semPower.powerCLPM()}}.  For hypotheses regarding longitudinal measurement invariance, see \code{\link[=semPower.powerLI]{semPower.powerLI()}}.

Beyond the arguments explicitly contained in the function call, additional arguments
are required specifying the factor model and the requested type of power analysis.

Additional arguments related to the \strong{definition of the factor model}:
\itemize{
\item \code{Lambda}: The factor loading matrix (with the number of columns equaling the number of factors).
\item \code{loadings}: Can be used instead of \code{Lambda}: Defines the primary loadings for each factor in a list structure, e. g. \code{loadings = list(c(.5, .4, .6), c(.8, .6, .6, .4))} defines a two factor model with three indicators loading on the first factor by .5, , 4., and .6, and four indicators loading on the second factor by .8, .6, .6, and .4.
\item \code{nIndicator}: Can be used instead of \code{Lambda}: Used in conjunction with \code{loadM}. Defines the number of indicators by factor, e. g., \code{nIndicator = c(3, 4)} defines a two factor model with three and four indicators for the first and second factor, respectively. \code{nIndicator} can also be a single number to define the same number of indicators for each factor.
\item \code{loadM}: Can be used instead of \code{Lambda}: Used in conjunction with \code{nIndicator}. Defines the loading either for all indicators (if a single number is provided) or separately for each factor (if a vector is provided), e. g. \code{loadM = c(.5, .6)} defines the loadings of the first factor to equal .5 and those of the second factor do equal .6.
}

So either \code{Lambda}, or \code{loadings}, or \code{nIndicator} and \code{loadM} need to be defined. Note that neither may contain the noise factors.
If the model contains observed variables only, use \code{Lambda = diag(x)} where \code{x} is the number of variables.

The order of the factors is (X1, X2, ..., X_nWaves).

Additional arguments related to the requested type of \strong{power analysis}:
\itemize{
\item \code{alpha}: The alpha error probability. Required for \code{type = 'a-priori'} and \code{type = 'post-hoc'}.
\item Either \code{beta} or \code{power}: The beta error probability and the statistical power (1 - beta), respectively. Only for \code{type = 'a-priori'}.
\item \code{N}: The sample size. Always required for \code{type = 'post-hoc'} and \code{type = 'compromise'}. For \code{type = 'a-priori'} and multiple group analysis, \code{N} is a list of group weights.
\item \code{abratio}: The ratio of alpha to beta. Only for \code{type = 'compromise'}.
}

If a \strong{simulated power analysis} (\code{simulatedPower = TRUE}) is requested, optional arguments can be provided as a list to \code{simOptions}:
\itemize{
\item \code{nReplications}: The targeted number of simulation runs. Defaults to 250, but larger numbers greatly improve accuracy at the expense of increased computation time.
\item \code{minConvergenceRate}:  The minimum convergence rate required, defaults to .5. The maximum actual simulation runs are increased by a factor of 1/minConvergenceRate.
\item \code{type}: specifies whether the data should be generated from a population assuming multivariate normality (\code{'normal'}; the default), or based on an approach generating non-normal data (\code{'IG'}, \code{'mnonr'}, \code{'RC'}, or \code{'VM'}).
The approaches generating non-normal data require additional arguments detailed below.
\item \code{missingVars}: vector specifying the variables containing missing data (defaults to NULL).
\item \code{missingVarProp}: can be used instead of \code{missingVars}: The proportion of variables containing missing data (defaults to zero).
\item \code{missingProp}: The proportion of missingness for variables containing missing data (defaults to zero), either a single value or a vector giving the probabilities for each variable.
\item \code{missingMechanism}: The missing data mechanism, one of \code{MCAR} (the default), \code{MAR}, or \code{NMAR}.
\item \code{nCores}: The number of cores to use for parallel processing. Defaults to 1 (= no parallel processing). This requires the \code{doSNOW} package.
}

\code{type = 'IG'} implements the independent generator approach (IG, Foldnes & Olsson, 2016) approach
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors. This requires the \code{covsim} package.

\code{type = 'mnonr'} implements the approach suggested by Qu, Liu, & Zhang (2020) and requires provision of  Mardia's multivariate skewness (\code{skewness})  and kurtosis  (\code{kurtosis}), where
skewness must be non-negative and kurtosis must be at least 1.641 skewness + p (p + 0.774), where p is the number of variables. This requires the \code{mnonr} package.

\code{type = 'RK'} implements the approach suggested by Ruscio & Kaczetow (2008) and requires provision of the population distributions
of each variable (\code{distributions}). \code{distributions} must be a list (if all variables shall be based on the same population distribution) or a list of lists.
Each component must specify the population distribution (e.g. \code{rchisq}) and additional arguments (\code{list(df = 2)}).

\code{type = 'VM'} implements the third-order polynomial method (Vale & Maurelli, 1983)
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors.  This requires the \code{semTools} package.
}
\examples{
\dontrun{
# Determine required N in a 10-wave ARMA model
# to detect that the autoregressive effects differ across waves
# with a power of 80\% on alpha = 5\%, where
# X is measured by 3 indicators loading by .5 each (at each wave), and
# the autoregressive effects vary between .5 and .7, and
# the moving average parameters are .3 at each wave and
# are assumed to be constant across waves (in both the H0 and the H1 model) and
# there are no lagged effects, and
# metric invariance and autocorrelated residuals are assumed
powerARMA <- semPower.powerARMA(
  'a-priori', alpha = .05, power = .80,
  nWaves = 10,
  autoregLag1 = c(.5, .7, .6, .5, .7, .6, .6, .5, .6),
  mvAvgLag1 = rep(.3, 9),
  variances = rep(1, 10),
  waveEqual = c('mvAvg'),
  nullEffect = 'autoreg',
  nIndicator = rep(3, 10), loadM = .5,
  invariance = TRUE, 
  autocorResiduals = TRUE
)

# show summary
summary(powerARMA)
# optionally use lavaan to verify the model was set-up as intended
lavaan::sem(powerARMA$modelH1, sample.cov = powerARMA$Sigma,
            sample.nobs = powerARMA$requiredN,
            sample.cov.rescale = FALSE)
lavaan::sem(powerARMA$modelH0, sample.cov = powerARMA$Sigma,
            sample.nobs = powerARMA$requiredN,
            sample.cov.rescale = FALSE)


# same as above, but determine power with N = 250 on alpha = .05
powerARMA <- semPower.powerARMA(
  'post-hoc', alpha = .05, N = 250,
  nWaves = 10,
  autoregLag1 = c(.5, .7, .6, .5, .7, .6, .6, .5, .6),
  mvAvgLag1 = rep(.3, 9),
  variances = rep(1, 10),
  waveEqual = c('mvAvg'),
  nullEffect = 'autoreg',
  nIndicator = rep(3, 10), loadM = .5,
  invariance = TRUE, 
  autocorResiduals = TRUE
)

# same as above, but determine the critical chi-square with N = 250 so that alpha = beta
powerARMA <- semPower.powerARMA(
  'compromise', abratio = 1, N = 250,
  nWaves = 10,
  autoregLag1 = c(.5, .7, .6, .5, .7, .6, .6, .5, .6),
  mvAvgLag1 = rep(.3, 9),
  variances = rep(1, 10),
  waveEqual = c('mvAvg'),
  nullEffect = 'autoreg',
  nIndicator = rep(3, 10), loadM = .5,
  invariance = TRUE, 
  autocorResiduals = TRUE
)
  
# same as above, but compare to the saturated model
# (rather than to the less restricted model)
powerARMA <- semPower.powerARMA(
  'a-priori', alpha = .05, power = .80, comparison = 'saturated',
  nWaves = 10,
  autoregLag1 = c(.5, .7, .6, .5, .7, .6, .6, .5, .6),
  mvAvgLag1 = rep(.3, 9),
  variances = rep(1, 10),
  waveEqual = c('mvAvg'),
  nullEffect = 'autoreg',
  nIndicator = rep(3, 10), loadM = .5,
  invariance = TRUE, 
  autocorResiduals = TRUE
)


# same as above, but assume only observed variables
powerARMA <- semPower.powerARMA(
  'a-priori', alpha = .05, power = .80,
  nWaves = 10,
  autoregLag1 = c(.5, .7, .6, .5, .7, .6, .6, .5, .6),
  mvAvgLag1 = rep(.3, 9),
  variances = rep(1, 10),
  waveEqual = c('mvAvg'),
  nullEffect = 'autoreg',
  Lambda = diag(1, 10),
  invariance = TRUE, 
  autocorResiduals = TRUE
)

# same as above, but provide reduced loadings matrix to define that
# X is measured by 3 indicators each loading by .5, .6, .4 (at each wave)
powerARMA <- semPower.powerARMA(
  'a-priori', alpha = .05, power = .80,
  nWaves = 10,
  autoregLag1 = c(.5, .7, .6, .5, .7, .6, .6, .5, .6),
  mvAvgLag1 = rep(.3, 9),
  variances = rep(1, 10),
  waveEqual = c('mvAvg'),
  nullEffect = 'autoreg',
  loadings = list(
    c(.5, .6, .4),  # X1
    c(.5, .6, .4),  # X2
    c(.5, .6, .4),  # X3
    c(.5, .6, .4),  # X4
    c(.5, .6, .4),  # X5
    c(.5, .6, .4),  # X6
    c(.5, .6, .4),  # X7
    c(.5, .6, .4),  # X8
    c(.5, .6, .4),  # X9
    c(.5, .6, .4)   # X10
  ),
  invariance = TRUE, 
  autocorResiduals = TRUE
)

# same as above, but detect that the moving average parameters differ across waves
# with a power of 80\% on alpha = 5\%, where
# the moving average parameters vary between .05 and .4, and
# the autoregressive effects are .5 at each wave and
# are assumed to be constant across waves (in both the H0 and the H1 model)
powerARMA <- semPower.powerARMA(
  'a-priori', alpha = .05, power = .80,
  nWaves = 10,
  autoregLag1 = rep(.5, 9),
  mvAvgLag1 = c(.1, .05, .2, .1, .1, .3, .4, .4, .4),
  variances = rep(1, 10),
  waveEqual = c('autoreg'),
  nullEffect = 'mvAvg',
  nIndicator = rep(3, 10), loadM = .5,
  invariance = TRUE, 
  autocorResiduals = TRUE
)


# same as above, but detect that the (noise) variances differ across waves
# with a power of 80\% on alpha = 5\%, where
# the variances vary between 0.5 and 2, and
# the autoregressive effects are .5 at each wave and
# the moving average parameters are .3 at each wave and
# bothj are assumed to be constant across waves (in both the H0 and the H1 model)
powerARMA <- semPower.powerARMA(
  'a-priori', alpha = .05, power = .80,
  nWaves = 10,
  autoregLag1 = rep(.5, 9),
  mvAvgLag1 = rep(.3, 9),
  variances = c(1, .5, .7, .6, .7, .9, 1.2, 1.7, 2.0, 1.5),
  waveEqual = c('autoreg', 'mvAvg'),
  nullEffect = 'var',
  nIndicator = rep(3, 10), loadM = .5,
  invariance = TRUE, 
  autocorResiduals = TRUE
)


# same as above, but include a meanstructure and
# detect that the means differ across waves
# with a power of 80\% on alpha = 5\%, where
# the means vary between 0 and .5, and
# the autoregressive effects are .5 at each wave and
# the moving average parameters are .3 at each wave and
# the variances are 1 at each wave and
# all are assumed to be constant across waves (in both the H0 and the H1 model) and
# metric and scalar invariance is assumed
powerARMA <- semPower.powerARMA(
  'a-priori', alpha = .05, power = .80,
  nWaves = 10,
  autoregLag1 = rep(.5, 9),
  mvAvgLag1 = rep(.3, 9),
  variances = rep(1, 10),
  means = c(0, .1, .2, .3, .4, .5, .3, .4, .5, .5),
  waveEqual = c('autoreg', 'mvAvg', 'var'),
  nullEffect = 'mean',
  nIndicator = rep(3, 10), loadM = .5,
  invariance = TRUE, 
  autocorResiduals = TRUE
)

# Determine required N in a 10-wave ARMA model
# to detect that the autoregressive lag-2 effects differ from zero
# with a power of 80\% on alpha = 5\%, where
# the lag-2 autoregressive effects are .2 at each wave and 
# the lag-2 autoregressive effects are .1 at each wave and
# the autoregressive effects are .5 at each wave and
# the moving average parameters are .3 at each wave and
# the noise variances are equal to 1 in each wave,
# and all are assumed to be constant across waves (in both the H0 and the H1 model) and
# metric invariance and autocorrelated residuals are assumed, and
# the autoregressive lag2- and lag3-effects are estimated 
powerARMA <- semPower.powerARMA(
  'a-priori', alpha = .05, power = .80,
  nWaves = 10,
  autoregLag1 = rep(.5, 9),
  autoregLag2 = rep(.2, 8),
  autoregLag3 = rep(.1, 7),
  mvAvgLag1 = rep(.3, 9),
  variances = rep(1, 10),
  waveEqual = c('mvAvg', 'autoreg', 'var', 'autoreglag2', 'autoreglag3'),
  nullEffect = 'autoreglag2 = 0',
  nIndicator = rep(3, 10), loadM = .5,
  invariance = TRUE, 
  autocorResiduals = TRUE
)

# similar as above, but get required N to detect that 
# lag-2 moving average parameters are constant across waves 
powerARMA <- semPower.powerARMA(
  'a-priori', alpha = .05, power = .80,
  nWaves = 10,
  autoregLag1 = rep(.5, 9),
  autoregLag2 = rep(.2, 8),
  mvAvgLag1 = rep(.3, 9),
  mvAvgLag2 = c(.1, .2, .3, .1, .2, .3, .1, .1),
  variances = rep(1, 10),
  waveEqual = c('mvAvg', 'autoreg', 'var', 'autoreglag2'),
  nullEffect = 'mvAvgLag2',
  nIndicator = rep(3, 10), loadM = .5,
  invariance = TRUE
)


# Determine required N in a 5-wave ARMA model
# to detect that the autoregressive effects in group 1
# differ from the ones in group 2, where
# both groups are equal-sized
# with a power of 80\% on alpha = 5\%, where
# X is measured by 3 indicators loading by .5 each (at each wave and in each group), and
# the autoregressive effects in group 1 are .5 (constant across waves) and
# the autoregressive effects in group 2 are .6 (constant across waves) and
# the moving average parameters are .25 at each wave and in both groups and
# the variances are 1 at each wave and in both groups and 
# all are assumed to be constant across waves (in both the H0 and the H1 model)
# metric invariance (across both waves and groups) and
# autocorrelated residuals are assumed
powerARMA <- semPower.powerARMA(
  'a-priori', alpha = .05, power = .80, N = list(1, 1),
  nWaves = 5,
  autoregLag1 = list(
    c(.5, .5, .5, .5),   # group 1
    c(.6, .6, .6, .6)),  # group 2
  mvAvgLag1 = rep(.25, 4),
  variances = rep(1, 5),
  waveEqual = c('autoreg', 'var', 'mvavg'),
  nullEffect = 'autoregA = autoregB',
  nIndicator = rep(3, 5), loadM = .5,
  invariance = TRUE, 
  autocorResiduals = TRUE
)


# Determine required N in a 5-wave ARMA model
# to detect that the means in group 1
# differ from the means in group 2, where
# both groups are equal-sized
# with a power of 80\% on alpha = 5\%, where
# X is measured by 3 indicators loading by .5 each (at each wave and in each group), and
# the autoregressive effects are .5 at each wave and in both groups and
# the moving average parameters are .25 at each wave and in both groups and
# the variances are 1 at each wave and in both groups and 
# all are assumed to be constant across waves (in both the H0 and the H1 model) and
# invariance of variances, autoregressive effects, and moving average parameters 
# across groups as well as
# metric and scalar invariance (across both waves and groups) and
# autocorrelated residuals are assumed
powerARMA <- semPower.powerARMA(
  'a-priori', alpha = .05, power = .80, N = list(1, 1),
  nWaves = 5,
  autoregLag1 = list(
    c(.5, .5, .5, .5),   # group 1
    c(.5, .5, .5, .5)),  # group 2
  mvAvgLag1 = rep(.25, 4),
  variances = rep(1, 5),
  means = list(
    c(0, .1, .1, .1, .1),  # group 1
    c(0, .4, .4, .4, .4)   # group 2
  ),
  waveEqual = c('autoreg', 'var', 'mvavg', 'mean'),
  groupEqual = c('var', 'autoreg', 'mvavg'),
  nullEffect = 'meanA = meanB',
  nIndicator = rep(3, 5), loadM = .5,
  invariance = TRUE, 
  autocorResiduals = TRUE
)

# perform a simulated post-hoc power analysis
# with 250 replications
set.seed(300121)
powerARMA <- semPower.powerARMA(
  'post-hoc', alpha = .05, N = 500,
  nWaves = 5,
  autoregLag1 = c(.3, .7, .6, .3),
  mvAvgLag1 = rep(.3, 4),
  variances = rep(1, 5),
  waveEqual = c('mvAvg'),
  nullEffect = 'autoreg',
  nIndicator = rep(3, 5), loadM = .5,
  invariance = TRUE, 
  autocorResiduals = TRUE, 
  simulatedPower = TRUE,
  simOptions = list(nReplications = 250)
)
}
}
\seealso{
\code{\link[=semPower.genSigma]{semPower.genSigma()}} \code{\link[=semPower.aPriori]{semPower.aPriori()}} \code{\link[=semPower.postHoc]{semPower.postHoc()}} \code{\link[=semPower.compromise]{semPower.compromise()}}
}
