% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/auxiliary.R
\name{weightsLinearInterp1D}
\alias{weightsLinearInterp1D}
\alias{weightsLinearInterp2D}
\title{Weights for linear interpolation in 1D and 2D}
\usage{
weightsLinearInterp1D(x, g1, g2, circular = FALSE)

weightsLinearInterp2D(x, y, gx1, gx2, gy1, gy2, circular = FALSE)
}
\arguments{
\item{x, y}{vectors of length \code{n} containing the points where the
weights should be computed.}

\item{g1, g2, gx1, gx2, gy1, gy2}{vectors of length \code{n} such that
\code{g1 <= x <= g2} for 1D and \code{gx1 <= x <= gx2} and
\code{gy1 <= y <= gy2} for 2D.}

\item{circular}{flag to indicate whether the differences should be
circularly wrapped.}
}
\value{
\itemize{
\item For 1D, a matrix of size \code{c(n, 2)} containing the weights for
lower (\code{g1}) and upper (\code{g1}) bins.
\item For 2D, a matrix of size \code{c(n, 4)} containing the weights for
lower-lower (\code{g1x}, \code{g1y}), \emph{upper-lower} (\code{g2x},
\code{g1y}), \emph{lower-upper} (\code{g1x}, \code{g2y}) and upper-upper
(\code{g2x}, \code{g2y}) bins. \code{cbind(g1x, g1y)},
\code{cbind(g1x, g1y)}, \code{cbind(g1x, g1y)} and \code{cbind(g2x, g2y)}.
}
}
\description{
Computation of weights for linear interpolation in 1D and 2D.
}
\details{
See the examples for how to use the weights for linear binning.
}
\examples{
# 1D, linear
x <- seq(-4, 4, by = 0.5)
g1 <- x - 0.25
g2 <- x + 0.5
w <- weightsLinearInterp1D(x = x, g1 = g1, g2 = g2)
f <- function(x) 2 * x + 1
rowSums(w * cbind(f(g1), f(g2)))
f(x)

# 1D, circular
x <- seq(-pi, pi, by = 0.5)
g1 <- toPiInt(x - 0.25)
g2 <- toPiInt(x + 0.5)
w <- weightsLinearInterp1D(x = x, g1 = g1, g2 = g2, circular = TRUE)
f <- function(x) 2 * sin(x) + 1
rowSums(w * cbind(f(g1), f(g2)))
f(x)

# 2D, linear
x <- seq(-4, 4, by = 0.5)
y <- 2 * x
gx1 <- x - 0.25
gx2 <- x + 0.5
gy1 <- y - 0.75
gy2 <- y + 0.25
w <- weightsLinearInterp2D(x = x, y = y, gx1 = gx1, gx2 = gx2,
                           gy1 = gy1, gy2 = gy2)
f <- function(x, y) 2 * x + 3 * y + 1
rowSums(w * cbind(f(gx1, gy1), f(gx2, gy1), f(gx1, gy2), f(gx2, gy2)))
f(x, y)

# 2D, circular
x <- seq(-pi, pi, by = 0.5)
y <- toPiInt(2 * x)
gx1 <- toPiInt(x - 0.25)
gx2 <- toPiInt(x + 0.5)
gy1 <- toPiInt(y - 0.75)
gy2 <- toPiInt(y + 0.25)
w <- weightsLinearInterp2D(x = x, y = y, gx1 = gx1, gx2 = gx2,
                           gy1 = gy1, gy2 = gy2, circular = TRUE)
f <- function(x, y) 2 * sin(x) + 3 * cos(y) + 1
rowSums(w * cbind(f(gx1, gy1), f(gx2, gy1), f(gx1, gy2), f(gx2, gy2)))
f(x, y)
}
\keyword{internal}
