% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/class-command-args.R
\docType{class}
\name{scribeCommandArgs-class}
\alias{scribeCommandArgs-class}
\alias{scribeCommandArgs}
\title{{scribe} command arguments}
\description{
Reference class object for managing command line arguments.
}
\details{
This class manages the command line argument inputs when passed via
the \link{Rscript} utility.  Take the simple script below which adds two
numbers, which we will save in an executable file called \code{add.R},

\if{html}{\out{<div class="sourceCode sh">}}\preformatted{#!/usr/bin/env Rscript

library(scribe)
ca <- command_args()
ca$add_argument("--value1", default = 0L)
ca$add_argument("--value2", default = 0L)
args <- ca$parse()
writeLines(args$value1 + args$value2)
}\if{html}{\out{</div>}}

When called by a terminal, we can pass arguments and return a function.

\if{html}{\out{<div class="sourceCode sh">}}\preformatted{add.R --value1 10 --value2 1
11
}\if{html}{\out{</div>}}

When testing, you can simulate command line arguments by passing them into
the \code{input} field. By default, this will grab values from
\code{\link[base:commandArgs]{base::commandArgs()}}, so use with the \link{Rscript} utility doesn't require
any extra steps.

Most methods are designed to return \code{.self}, or the \link{scribeCommandArgs}
class. The exceptions to these are the the \verb{$get_*()} methods, which return
their corresponding values, and \verb{$parse()} which returns a named \code{list} of
the parsed input values.
}
\section{Fields}{

\describe{
\item{\code{input}}{\verb{[character]}\cr A character vector of command line arguments.
See also \code{\link[=command_args]{command_args()}}}

\item{\code{values}}{\verb{[list]}\cr A named \code{list} of values.  Empty on initialization
and populated during argument resolving.}

\item{\code{args}}{\verb{[list]}\cr a List of \link{scribeArg}s}

\item{\code{description}}{\verb{[character]}\cr Additional help information}

\item{\code{included}}{\verb{[character]}\cr Default \link{scribeArg}s to include}

\item{\code{examples}}{\verb{[character]}\cr Examples to print with help}

\item{\code{comments}}{\verb{[character]}\cr Comments printed with}

\item{\code{resolved}}{\verb{[logical]}\cr A \code{logical} value indicated if the
\verb{$resolve()} method has been successfully executed.}

\item{\code{working}}{\verb{[character]}\cr A copy of \code{input}.  Note: this is used to
track parsing progress and is not meant to be accessed directly.}

\item{\code{stop}}{\verb{[character]}\cr Determines parsing}
}}

\section{Methods}{

\describe{
\item{\code{add_argument(
  ...,
  action = arg_actions(),
  options = NULL,
  convert = default_convert,
  default = NULL,
  n = NA_integer_,
  info = NULL,
  execute = invisible
)}}{Add a \link{scribeArg} to \code{args}

    \describe{
      \item{\code{...}}{Either aliases or a \link{scribeArg}.  If the latter,
        all other arguments are ignored.  Note that only the first value
        (\link{..1}) is used.}
      \item{\code{action}, \code{options}, \code{convet}, \code{default},
        \code{n}, \code{info}}{See \code{\link[=new_arg]{new_arg()}}}
    }}

\item{\code{add_description(..., sep = "")}}{Add a value to \code{description}

    \describe{
      \item{\code{...}}{Information to paste into the description}
      \item{\code{sep}}{\code{character} separate for \code{...}}
    }}

\item{\code{add_example(x, comment = "", prefix = "$ ")}}{Add a value to \code{examples}

    \describe{
      \item{\code{x}}{A code example as a \code{character}}
      \item{\code{comment}}{An optional comment to append}
      \item{\code{prefix}}{An optional prefix for the example}
    }}

\item{\code{get_args(included = TRUE)}}{Retrieve \code{args}

    \describe{
      \item{\code{included}}{If \code{TRUE} also returns included default
        \link{scribeArg}s defined in \code{$initialize()}}
    }}

\item{\code{get_description()}}{Retrieve \code{description}}

\item{\code{get_examples()}}{Retrieve \code{examples}}

\item{\code{get_input()}}{Retrieve \code{input}}

\item{\code{get_values()}}{Retrieve \code{values}}

\item{\code{help()}}{Print the help information}

\item{\code{initialize(input = "", include = c("help", "version", NA_character_))}}{Initialize the \link{scribeCommandArgs} object.  The wrapper
    \code{\link[=command_args]{command_args()}} is recommended rather than
    calling this method directly.

    \describe{
      \item{\code{input}}{A \code{character} vector of command line arguments
        to parse}
      \item{\code{include}}{A character vector denoting which default
        \link{scribeArg}s to include in \code{args}}
    }}

\item{\code{parse()}}{Return a named \code{list} of parsed values of from each \link{scribeArg}
in \code{args}}

\item{\code{resolve()}}{Resolve the values of each \link{scribeArg} in \code{args}.  This method
is called prior to $parse()}

\item{\code{set_description(..., sep = "")}}{Set the value of \code{description}

    \describe{
      \item{\code{...}}{Information to paste into the description}
      \item{\code{sep}}{\code{character} separate for \code{...}}
    }}

\item{\code{set_example(x = character(), comment = "", prefix = "$ ")}}{Set the value of \code{examples}

    \describe{
      \item{\code{x}}{A code example as a \code{character}}
      \item{\code{comment}}{An optional comment to append}
      \item{\code{prefix}}{An optional prefix for the example}
    }}

\item{\code{set_input(value)}}{Set \code{input}.  Note: when called, \code{resolved} is (re)set to
     \code{FALSE} and values need to be parsed again.

    \describe{
      \item{\code{value}}{Value to set}
    }}

\item{\code{set_values(i = TRUE, value)}}{Set \code{values}

    \describe{
      \item{\code{i}}{Index value of \code{working} to set}
      \item{\code{value}}{The value to set}
    }}

\item{\code{version()}}{Print the \link{scribe-package} version}
}}

\examples{
# command_args() is recommended over direct use of scribeCommandArgs$new()

ca <- command_args(c(1, 2, 3, "--verbose"))
ca$add_argument("--verbose", action = "flag")
ca$add_argument("...", "values", info = "values to add", default = 0.0)
args <- ca$parse()

if (args$verbose) {
  message("Adding ", length(args$values), " values")
}

sum(args$values)

# $parse() returns a named list, which means scribeCommandArgs can function
# as a wrapper for calling R functions inside Rscript

ca <- command_args(c("mean", "--size", 20, "--absolute"))
ca$add_argument("fun", action = "list")
ca$add_argument("--size", default = 5L)
ca$add_argument("--absolute", action = "flag")
args <- ca$parse()

my_function <- function(fun, size, absolute = FALSE) {
  fun <- match.fun(fun)
  x <- sample(size, size, replace = TRUE)
  res <- fun(x)
  if (absolute) res <- abs(res)
  res
}

do.call(my_function, args)
}
\seealso{
Other scribe: 
\code{\link{command_args}()},
\code{\link{new_arg}()},
\code{\link{scribeArg-class}}
}
\concept{scribe}
