% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/averageSP.R, R/dbaSP.R
\name{averageSP}
\alias{averageSP}
\alias{dbaSP}
\title{Average a group of snow profiles}
\usage{
averageSP(
  SPx,
  n = 5,
  sm = summary(SPx),
  progressbar = require("progress", quietly = TRUE, character.only = TRUE),
  progressbar_pretext = NULL,
  classifyPWLs = list(pwl_gtype = c("SH", "DH")),
  classifyCRs = list(pwl_gtype = c("MFcr", "IF", "IFsc", "IFrc")),
  proportionPWL = 0.5,
  breakAtSim = 0.9,
  breakAfter = 2,
  verbose = FALSE,
  ...
)

dbaSP(
  SPx,
  Avg,
  sm = summary(SPx),
  resamplingRate = 0.5,
  proportionPWL = 0.3,
  maxiter = 10,
  breakAtSim = 0.99,
  breakAfter = 1,
  plotChanges = FALSE,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{SPx}{SPx a \link{snowprofileSet} object. Note that the profile layers need to contain a column
called \verb{$layerOfInterest} which classifies weak layers. While \link{averageSP} will label weak layers automatically if not done by the user beforehand, \link{dbaSP} won't do that but fail instead!;
consider thinking about how you want to label weak layers, see Description, \code{classifyPWLs} below, and the references.
Also note, that if you wish to average the \emph{rescaled} profile set, do so manually before calling this function (see examples).}

\item{n}{the number of initial conditions that will be used to run \link{dbaSP}; see also \link{chooseICavg}.}

\item{sm}{a \link{summary} of \code{SPx} metadata}

\item{progressbar}{should a progressbar be displayed (the larger n, the more meaningful the progressbar)}

\item{progressbar_pretext}{a character string to be prepended to the progressbar (mainly used by higher level cluster function)}

\item{classifyPWLs}{an argument list for a function call to \link[sarp.snowprofile:findPWL]{sarp.snowprofile::findPWL} which returns relevant PWLs for identifying initial conditions. \strong{Importantly}, these arguments will also be used
to label weak layers in the profiles, if these labels do not yet exist in the layers objects as column \verb{$layerOfInterest}.
Check out the documentation of \code{findPWL} to familiarize yourself with your manifold options!}

\item{classifyCRs}{an argument list for a function call to \link[sarp.snowprofile:findPWL]{sarp.snowprofile::findPWL} which returns relevant crusts for identifying initial conditions.}

\item{proportionPWL}{decimal number that specifies the proportion required to average an ensemble of grain types as weak layer type.
A value of 0.3, for example, means that layers will get averaged to a PWL type if 30\% of the layers are of PWL type.
Meaningful range is between \verb{[0.1, 0.5]}. Values larger than 0.5 get set to 0.5.}

\item{breakAtSim}{stop iterations when \link{simSP} between the last average profiles is beyond that value. Can range between \verb{[0, 1]}. Default values differ between \link{dbaSP} and \link{averageSP}.}

\item{breakAfter}{integer specifying how many values of simSP need to be above \code{breakAtSim} to stop iterating. Default values differ between \link{dbaSP} and \link{averageSP}.}

\item{verbose}{print similarities between old and new average in between iterations?}

\item{...}{alignment configurations which are passed on to \link{dbaSP} and then further to \link{dtwSP}. Note, that you can't provide \code{rescale2refHS}, which is always set to FALSE. If you wish to rescale
the profiles, read the description of the \code{SPx} parameter and the examples.}

\item{Avg}{the initial average snow profile: either a \link{snowprofile} object or an index to an initial average profile in SPx}

\item{resamplingRate}{Resampling rate for a regular depth grid among the profiles}

\item{maxiter}{maximum number of iterations}

\item{plotChanges}{specify whether and how you want to plot the dba process: either \code{FALSE}, 'TRUE\code{==}'iterations'\verb{, or }'averages+last'`}
}
\value{
A list of class \code{avgSP} that contains the fields
\itemize{
\item \verb{$avg}: the resulting average profile
\item \verb{$set}: the corresponding resampled profiles of the group
\item \verb{$call}: (only with \code{averageSP}) the function call
\item \verb{$prelabeledPWLs}: (only with \code{averageSP}) boolean scalar whether PWLs (or any other layers of interest) were prelabeled before this routine (\code{TRUE}) or labeled by this routine (\code{FALSE})
}
}
\description{
The functions \link{dbaSP} and \link{averageSP} implement Dynamic Time Warping Barycenter Averaging of snow profiles.
The convenient wrapper \link{averageSP} takes care of choosing several appropriate initial conditions and picking the optimal end result (by minimizing the mean squared error
between the average profile and the profile set). To pay appropriate attention to (thin) weak layers, weak layers need to be labeled in the profiles.
You can either do that manually before calling this routine to suit your personal needs, or you can provide specific properties (in \code{classifyPWLs})
so that weak layers be labeled according to these properties by \link[sarp.snowprofile:findPWL]{sarp.snowprofile::labelPWL}.
For more details, refer to the reference paper.
}
\section{Functions}{
\itemize{
\item \code{averageSP}: convenient wrapper function

\item \code{dbaSP}: DTW barycenter averaging of snow profiles (low level worker function)
}}

\examples{
## EXAMPLES OF averageSP
this_example_runs_about_10s <- TRUE
if (!this_example_runs_about_10s) {  # exclude from cran checks

## compute the average profile of the demo object 'SPgroup'
## * by labeling SH/DH layers as weak layers,
##   - choosing 3 initial conditions with an above average number of weak layers
##   - in as many depth ranges as possible
## * and neglecting crusts for initial conditions

  avgList <- averageSP(SPgroup, n = 3,
                       classifyPWLs = list(pwl_gtype = c("SH", "DH")),
                       classifyCRs = NULL)

  opar <- par(mfrow = c(1, 2))
  plot(avgList$avg, ymax = max(summary(avgList$set)$hs))
  plot(avgList$set, SortMethod = "unsorted", xticklabels = "originalIndices")
  par(opar)


  ## compute the average profile of the demo object 'SPgroup'
## * by labeling SH/DH/FC/FCxr layers with an RTA threshold of 0.65 as weak layers,
## * otherwise as above

  SPx <- snowprofileSet(lapply(SPgroup, computeRTA))
  avgList <- averageSP(SPx, n = 3,
                       classifyPWLs = list(pwl_gtype = c("SH", "DH", "FC", "FCxr"),
                                           threshold_RTA = 0.65),
                       classifyCRs = NULL)

  opar <- par(mfrow = c(1, 2))
  plot(avgList$avg, ymax = max(summary(avgList$set)$hs))
  plot(avgList$set, SortMethod = "unsorted", xticklabels = "originalIndices")
  par(opar)

}



## EXAMPLES OF dbaSP
## either rescale profiles beforehand...
if (FALSE) {  # don't run in package check to save time
  SPx <- reScaleSampleSPx(SPgroup)$set          # rescale profiles
  SPx <- snowprofileSet(lapply(SPx, labelPWL))  # label PWLs
  DBA <- dbaSP(SPx, 5, plotChanges = TRUE)      # average profiles
}

## or use unscaled snow heights:
if (FALSE) {  # don't run in package check to save time
  SPx <- snowprofileSet(lapply(SPgroup, labelPWL))  # label PWLs
  DBA <- dbaSP(SPx, 5, plotChanges = TRUE)          # average profiles
}
}
\references{
Herla, F., Haegeli, P., and Mair, P.: Brief communication: A numerical tool for averaging large data sets of snow
stratigraphy profiles useful for avalanche forecasting, The Cryosphere Discuss., https://doi.org/10.5194/tc-2022-29, in review, 2022.
}
\seealso{
\link{averageSPalongSeason}
}
\author{
fherla
}
