#' @title Encrypt a R object
#'
#' @description \code{encrypt_object} encrypts a R object as a raw object or a
#'   string and \code{decrypt_object} decrypts a raw object or a
#'   string(encrypted by \code{encrypt_object})
#'
#' @param object A R object to be encrypted
#' @param ascii \code{TRUE} if the R object is to be encrypted as a string.
#'   Default is \code{FALSE}
#' @param key A string without embbeded NULL. Default is 'pass'
#' @param method Currently, a stub. It should be 'symmetric'(default)
#'
#' @return A raw object if \code{ascii} is \code{FALSE}. A string if
#'   \code{ascii} is \code{TRUE}.
#'
#' @examples
#' temp <- encrypt_object(1:3)
#' all(
#'   is.raw(temp)
#'   , decrypt_object(temp) == 1:3)
#'
#' temp <- encrypt_object(iris, ascii = TRUE)
#' all(
#'   is.character(temp)
#'   , decrypt_object(temp) == iris
#'   , identical(decrypt_object(temp), iris))
#' rm(temp)
#'
#' @export
#'
encrypt_object  <- function(object
                            , ascii  = FALSE
                            , key    = "pass"
                            , method = "symmetric"){
  assert_that(!missing(object))
  assert_that(is.flag(ascii))
  assert_that(is.string(key))
  assert_that(method %in% c("symmetric"))

  raw_object <- try(serialize(object, NULL), silent = TRUE)
  if(is.error(raw_object)){
    stop("Unable to serialize the object")
  }

  keyAsRaw <- try(hash(charToRaw(key)), silent = TRUE)
  if(is.error(keyAsRaw)){
    stop("Unable to convert 'key' into raw. Possibly encountered an embedded NULL.")
  }

  if(method == "symmetric"){
    object_enc_raw <- data_encrypt(raw_object
                                   , keyAsRaw
                                   , hash(charToRaw("nounce"), size = 24)
                                   )
  }

  attributes(object_enc_raw) <- NULL
  if(ascii){
    return(base64encode(object_enc_raw))
  } else {
    return(object_enc_raw)
  }
}

#' @title Decrypt an encrypted R object
#'
#' @description \code{encrypt_object} encrypts a R object as a raw object or a
#'   string and \code{decrypt_object} decrypts a raw object or a
#'   string(encrypted by \code{encrypt_object})
#'
#' @param encryptedObject A raw object or a string to be decrypted
#' @param key A string without embbeded NULL. Default is 'pass'.
#' @param method Currently, a stub. It should be 'symmetric'(default)
#'
#' @return A R object
#'
#' @examples
#' temp <- encrypt_object(1:3)
#' all(
#'   is.raw(temp)
#'   , decrypt_object(temp) == 1:3)
#'
#' temp <- encrypt_object(iris, ascii = TRUE)
#' all(
#'   is.character(temp)
#'   , decrypt_object(temp) == iris
#'   , identical(decrypt_object(temp), iris))
#' rm(temp)
#'
#' @export
#'
decrypt_object  <- function(encryptedObject
                            , key    = "pass"
                            , method = "symmetric"){
  assert_that(is.string(key))
  assert_that(method %in% c("symmetric"))
  assert_that(class(encryptedObject) == "raw" || is.string(encryptedObject))

  if(is.string(encryptedObject)){
    encryptedObject <- try(base64decode(encryptedObject), silent = TRUE)
    if(is.error(encryptedObject)){
      stop("Unable to Decrypt. Ensure that 'encryptedObject' was the result of 'encypt_object' or 'serialize2' function with param 'ascii' set to TRUE.")
    }
  }

  keyAsRaw <- try(hash(charToRaw(key)), silent = TRUE)
  if(is.error(keyAsRaw)){
    stop("Unable to convert 'key' into raw. Possibly encountered an embedded NULL.")
  }

  if(method == "symmetric"){
    object_dec_raw <-
      try(data_decrypt(encryptedObject
                       , keyAsRaw
                       , hash(charToRaw("nounce"), size = 24)
                       )
          , silent = TRUE
          )
  }

  if(is.error(object_dec_raw)){
    stop("Unable to Decrypt. Ensure that input was generated by 'encypt_object' or 'serialize2' function. Ensure that 'key' and 'method' are correct.")
  }
  return(unserialize(object_dec_raw))
}
