% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rSVD.R
\name{rsvd}
\alias{rsvd}
\title{Randomized Singular Value Decomposition .}
\usage{
rsvd(A, k = NULL, nu = NULL, nv = NULL, p = 5, q = 2,
  method = "standard", sdist = "unif", vt = FALSE)
}
\arguments{
\item{A}{array_like \cr
a real/complex input matrix (or data frame), with dimensions \eqn{(m, n)}.}

\item{k}{int, optional \cr
determines the target rank of the low-rank decomposition and should satisfy \eqn{k << min(m,n)}.}

\item{nu}{int, optional \cr
the number of left singular vectors to be computed. This must be between \eqn{0}
and \eqn{k}.}

\item{nv}{int, optional \cr
the number of right singular vectors to be computed. This must be between \eqn{0}
and \eqn{k}.}

\item{p}{int, optional \cr
oversampling parameter for (default \eqn{p=5}).}

\item{q}{int, optional \cr
number of power iterations (default \eqn{q=2}).}

\item{method}{str c('standard', 'fast'), optional \cr
'standard' : (default): Standard algorithm as described in [1, 2]. \cr
'fast' : Version II algorithm as described in [2].}

\item{sdist}{str c('normal', 'unif'), optional \cr
Specifies the sampling distribution. \cr
'unif' : (default) Uniform `[-1,1]`. \cr
'normal' : Normal `~N(0,1)`. \cr}

\item{vt}{bool (\eqn{TRUE}, \eqn{FALSE}), optional \cr
\eqn{TRUE} : returns the transposed right singular vectors \eqn{vt}. \cr
\eqn{FALSE} : (default) returns the right singular vectors as \eqn{v}, this is the format
as \code{\link{svd}} returns \eqn{v}.}

\item{.............}{.}
}
\value{
\code{rsvd} returns a list containing the following three components:
\item{u}{  array_like \cr
          Right singular values; array with dimensions \eqn{(m, k)}.
}
\item{d}{  array_like \cr
          Singular values; 1-d array of length \eqn{(k)}.
}
\item{v (or vt)}{  array_like \cr
          Left singular values; array with dimensions \eqn{(n, k)}. \cr
          Or if \eqn{vt=TRUE}, array with dimensions \eqn{(k, n)}.
}
\item{.............}{.}
}
\description{
Compute the approximate low-rank singular value decomposition (SVD) of a rectangular matrix.
}
\details{
The singular value decomposition (SVD) plays a central role in data analysis and scientific computing.
Randomized SVD (rSVD) is a fast algorithm to compute the the approximate
low-rank SVD of a rectangular \eqn{(m,n)} matrix \eqn{A}
using a probablistic algorithm.
Given a target rank \eqn{k << n}, \code{rsvd} factors the input matrix \eqn{A} as
\eqn{A = U * diag(d) * V'}. The right singluar vectors are the columns of the
real or complex unitary matrix \eqn{U} . The left singular vectors are the columns
of the real or complex unitary matrix \eqn{V}. The singular values \eqn{d} are
non-negative and real numbers.

The parameter \eqn{p} is a oversampling parameter to improve the approximation.
A value between 2 and 10 is recommended and \eqn{p=5} is set as default.

The parameter \eqn{q} specifies the number of normalized power iterations
(subspace iterations) to reduce the approximation error. This is recommended
if the the singular values decay slowly. In practice 1 or 2 iterations
archive good results, however, computing power iterations increases the
computational time. The number of power iterations is set to \eqn{q=2} by default.

If \eqn{k > (min(n,m)/1.5)}, a deterministic partial or truncated \code{\link{svd}}
algorithm might be faster.
}
\note{
The singular vectors are not unique and only defined up to sign
(a constant of modulus one in the complex case). If a left singular vector
has its sign changed, changing the sign of the corresponding right vector
gives an equivalent decomposition.
}
\examples{
library(rsvd)
set.seed(123)

#Create real random test matrix with dimension (m, n) and rank k
m = 10
n = 3
k = 5
A <- matrix(runif(m*k), m, k)
A <- A \%*\% t(A)
A <- A[,1:n]

#Randomized SVD, no low-rank approximation
s <- rsvd(A)
Atilde = s$u \%*\% diag(s$d) \%*\% t(s$v)
100 * norm( A - Atilde, 'F') / norm( Atilde,'F') #Percentage reconstruction error << 1e-8

#Randomized SVD, low-rank approximation k=3
s <- rsvd(A, k=3)
Atilde = s$u \%*\% diag(s$d) \%*\% t(s$v)
100 * norm( A - Atilde, 'F') / norm( Atilde,'F') #Percentage reconstruction error << 1e-8

#Randomized SVD, low-rank approximation k=2
s <- rsvd(A, k=2)
Atilde = s$u \%*\% diag(s$d) \%*\% t(s$v)
100 * norm( A - Atilde, 'F') / norm( Atilde,'F') #Percentage reconstruction error < 3.5\%


}
\author{
N. Benjamin Erichson, \email{nbe@st-andrews.ac.uk}
}
\references{
\itemize{
  \item  [1] N. Halko, P. Martinsson, and J. Tropp.
         "Finding structure with randomness: probabilistic
         algorithms for constructing approximate matrix
         decompositions" (2009).
         (available at arXiv \url{http://arxiv.org/abs/0909.4061}).
  \item  [2] S. Voronin and P.Martinsson.
         "RSVDPACK: Subroutines for computing partial singular value
         decompositions via randomized sampling on single core, multi core,
         and GPU architectures" (2015).
         (available at `arXiv \url{http://arxiv.org/abs/1502.05366}).
}
}
\seealso{
\code{\link{svd}}, \code{\link{rpca}}
}

