% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/posterior_predict.R
\name{posterior_predict.stanreg}
\alias{posterior_predict}
\alias{posterior_predict.stanreg}
\title{Draw from posterior predictive distribution}
\usage{
\method{posterior_predict}{stanreg}(object, newdata = NULL, draws = NULL,
  re.form = NULL, fun = NULL, seed = NULL, offset = NULL, ...)
}
\arguments{
\item{object}{A fitted model object returned by one of the 
\pkg{rstanarm} modeling functions. See \code{\link{stanreg-objects}}.}

\item{newdata}{Optionally, a data frame in which to look for variables with
which to predict. If omitted, the model matrix is used. If \code{newdata}
is provided and any variables were transformed (e.g. rescaled) in the data
used to fit the model, then these variables must also be transformed in
\code{newdata}. This only applies if variables were transformed before
passing the data to one of the modeling functions and \emph{not} if
transformations were specified inside the model formula. Also see the Note
section below for a note about using the \code{newdata} argument with with
binomial models.}

\item{draws}{An integer indicating the number of draws to return. The default
and maximum number of draws is the size of the posterior sample.}

\item{re.form}{If \code{object} contains \code{\link[=stan_glmer]{group-level}}
parameters, a formula indicating which group-level parameters to
condition on when making predictions. \code{re.form} is specified in the
same form as for \code{\link[lme4]{predict.merMod}}. The default,
\code{NULL}, indicates that all estimated group-level parameters are
conditioned on. To refrain from conditioning on any group-level parameters,
specify \code{NA} or \code{~0}. The \code{newdata} argument may include new
\emph{levels} of the grouping factors that were specified when the model
was estimated, in which case the resulting posterior predictions
marginalize over the relevant variables.}

\item{fun}{An optional function to apply to the results. \code{fun} is found
by a call to \code{\link{match.fun}} and so can be specified as a function
object, a string naming a function, etc.}

\item{seed}{An optional \code{\link[=set.seed]{seed}} to use.}

\item{offset}{A vector of offsets. Only required if \code{newdata} is
specified and an \code{offset} argument was specified when fitting the
model.}

\item{...}{Currently ignored.}
}
\value{
A \code{draws} by \code{nrow(newdata)} matrix of simulations from the
  posterior predictive distribution. Each row of the matrix is a vector of 
  predictions generated using a single draw of the model parameters from the 
  posterior distribution. The returned matrix will also have class
  \code{"ppd"} to indicate it contains draws from the posterior predictive
  distribution.
}
\description{
The posterior predictive distribution is the distribution of the outcome
implied by the model after using the observed data to update our beliefs
about the unknown parameters in the model. Simulating data from the posterior
predictive distribution using the observed predictors is useful for checking
the fit of the model. Drawing from the posterior predictive distribution at
interesting values of the predictors also lets us visualize how a
manipulation of a predictor affects (a function of) the outcome(s). With new
observations of predictor variables we can use the posterior predictive
distribution to generate predicted outcomes.
}
\note{
For binomial models with a number of trials greater than one (i.e., not
  Bernoulli models), if \code{newdata} is specified then it must include all
  variables needed for computing the number of binomial trials to use for the
  predictions. For example if the left-hand side of the model formula is
  \code{cbind(successes, failures)} then both \code{successes} and
  \code{failures} must be in \code{newdata}. The particular values of
  \code{successes} and \code{failures} in \code{newdata} do not matter so
  long as their sum is the desired number of trials. If the left-hand side of
  the model formula were \code{cbind(successes, trials - successes)} then
  both \code{trials} and \code{successes} would need to be in \code{newdata},
  probably with \code{successes} set to \code{0} and \code{trials} specifying
  the number of trials. See the Examples section below and the
  \emph{How to Use the rstanarm Package} for examples.
}
\examples{
if (!exists("example_model")) example(example_model)
yrep <- posterior_predict(example_model)
table(yrep)

\donttest{
# Using newdata
counts <- c(18,17,15,20,10,20,25,13,12)
outcome <- gl(3,1,9)
treatment <- gl(3,3)
fit3 <- stan_glm(counts ~ outcome + treatment, family = poisson(link="log"),
                prior = normal(0, 1), prior_intercept = normal(0, 5))
nd <- data.frame(treatment = factor(rep(1,3)), outcome = factor(1:3))
ytilde <- posterior_predict(fit3, nd, draws = 500)
print(dim(ytilde))  # 500 by 3 matrix (draws by nrow(nd))
ytilde <- data.frame(count = c(ytilde),
                     outcome = rep(nd$outcome, each = 500))
ggplot2::ggplot(ytilde, ggplot2::aes(x=outcome, y=count)) +
  ggplot2::geom_boxplot() +
  ggplot2::ylab("predicted count")


# Using newdata with a binomial model.
# example_model is binomial so we need to set
# the number of trials to use for prediction.
# This could be a different number for each
# row of newdata or the same for all rows.
# Here we'll use the same value for all.
nd <- lme4::cbpp
print(formula(example_model))  # cbind(incidence, size - incidence) ~ ...
nd$size <- max(nd$size) + 1L   # number of trials
nd$incidence <- 0  # set to 0 so size - incidence = number of trials
ytilde <- posterior_predict(example_model, newdata = nd)


# Using fun argument to transform predictions
mtcars2 <- mtcars
mtcars2$log_mpg <- log(mtcars2$mpg)
fit <- stan_glm(log_mpg ~ wt, data = mtcars2)
ytilde <- posterior_predict(fit, fun = exp)
}

}
\seealso{
\code{\link{pp_check}} for graphical posterior predictive checks.
  Examples of posterior predictive checking can also be found in the
  \pkg{rstanarm} vignettes and demos.

\code{\link{predictive_error}} and \code{\link{predictive_interval}}.
}

