% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/isd.R
\name{isd}
\alias{isd}
\title{Get and parse NOAA ISD/ISH data}
\usage{
isd(usaf, wban, year, overwrite = TRUE, cleanup = TRUE,
  additional = TRUE, parallel = FALSE, cores = getOption("cl.cores",
  2), progress = FALSE, force = FALSE, ...)
}
\arguments{
\item{usaf, wban}{(character) USAF and WBAN code. Required}

\item{year}{(numeric) One of the years from 1901 to the current year.
Required.}

\item{overwrite}{(logical) To overwrite the path to store files in or not,
Default: \code{TRUE}}

\item{cleanup}{(logical) If \code{TRUE}, remove compressed \code{.gz} file
at end of function execution. Processing data takes up a lot of time, so we
cache a cleaned version of the data. Cleaning up will save you on disk
space. Default: \code{TRUE}}

\item{additional}{(logical) include additional and remarks data sections
in output. Default: \code{TRUE}. Passed on to
\code{\link[isdparser:isd_parse]{isdparser::isd_parse()}}}

\item{parallel}{(logical) do processing in parallel. Default: \code{FALSE}}

\item{cores}{(integer) number of cores to use: Default: 2. We look in
your option "cl.cores", but use default value if not found.}

\item{progress}{(logical) print progress - ignored if \code{parallel=TRUE}.
The default is \code{FALSE} because printing progress adds a small bit of
time, so if processing time is important, then keep as \code{FALSE}}

\item{force}{(logical) force download? Default: \code{FALSE}
We use a cached version (an .rds compressed file) if it exists, but
this will override that behavior.}

\item{...}{Curl options passed on to \link[crul:verb-GET]{crul::verb-GET}}
}
\value{
A tibble (data.frame).
}
\description{
Get and parse NOAA ISD/ISH data
}
\details{
\code{isd} saves the full set of weather data for the queried
site locally in the directory specified by the \code{path} argument. You
can access the path for the cached file via \code{attr(x, "source")}

We use \pkg{isdparser} internally to parse ISD files. They are
relatively complex to parse, so a separate package takes care of that.

This function first looks for whether the data for your specific
query has already been downloaded previously in the directory given by
the \code{path} parameter. If not found, the data is requested form NOAA's
FTP server. The first time a dataset is pulled down we must a) download the
data, b) process the data, and c) save a compressed .rds file to disk. The
next time the same data is requested, we only have to read back in the
.rds file, and is quite fast. The benfit of writing to .rds files is that
data is compressed, taking up less space on your disk, and data is read
back in quickly, without changing any data classes in your data, whereas
we'd have to jump through hoops to do that with reading in csv. The
processing can take quite a long time since the data is quite messy and
takes a bunch of regex to split apart text strings. We hope to speed
this process up in the future. See examples below for different behavior.
}
\note{
There are now no transformations (scaling, class changes, etc.)
done on the output data. This may change in the future with parameters
to toggle transformations, but none are done for now. See
\code{\link[isdparser:isd_transform]{isdparser::isd_transform()}} for transformation help.
Comprehensive transformations for all variables are not yet available
but should be available in the next version of this package.
}
\section{Errors}{

Note that when you get an error similar to \code{Error: download failed for ftp://ftp.ncdc.noaa.gov/pub/data/noaa/1955/011490-99999-1955.gz}, the
file does not exist on NOAA's ftp servers. If your internet is down,
you'll get a different error.
}

\section{File storage}{

We use \pkg{rappdirs} to store files, see
\code{\link[rappdirs:user_cache_dir]{rappdirs::user_cache_dir()}} for how we determine the directory on
your machine to save files to, and run
\code{rappdirs::user_cache_dir("rnoaa/isd")} to get that directory.

Note that cached files will continue to be used until they are deleted.
It's possible to run into problems when changes happen in your R
setup. For example, at least one user reported changing versions
of this package and running into problems because a cached data
file from a previous version of rnoaa did not work with the newer
version of rnoaa. You may want to occassionally delete all cached
files.
}

\examples{
\dontrun{
# Get station table
(stations <- isd_stations())

## plot stations
### remove incomplete cases, those at 0,0
df <- stations[complete.cases(stations$lat, stations$lon), ]
df <- df[df$lat != 0, ]
### make plot
library("leaflet")
leaflet(data = df) \%>\%
  addTiles() \%>\%
  addCircles()

# Get data
(res <- isd(usaf='011490', wban='99999', year=1986))
(res <- isd(usaf='011690', wban='99999', year=1993))
(res <- isd(usaf='109711', wban=99999, year=1970))

# "additional" and "remarks" data sections included by default
# can toggle that parameter to not include those in output, saves time
(res1 <- isd(usaf='011490', wban='99999', year=1986, force = TRUE))
(res2 <- isd(usaf='011490', wban='99999', year=1986, force = TRUE,
  additional = FALSE))

# The first time a dataset is requested takes longer
system.time( isd(usaf='782680', wban='99999', year=2011) )
system.time( isd(usaf='782680', wban='99999', year=2011) )

# Plot data
## get data for multiple stations
res1 <- isd(usaf='011690', wban='99999', year=1993)
res2 <- isd(usaf='782680', wban='99999', year=2011)
res3 <- isd(usaf='008415', wban='99999', year=2016)
res4 <- isd(usaf='109711', wban=99999, year=1970)
## combine data
library(dplyr)
res_all <- bind_rows(res1, res2, res3, res4)
# add date time
library("lubridate")
dd <- sprintf('\%s \%s', as.character(res_all$date), res_all$time)
res_all$date_time <- ymd_hm(dd)
## remove 999's
res_all <- filter(res_all, temperature < 900)

## plot
if (interactive()) {
  library(ggplot2)
  ggplot(res_all, aes(date_time, temperature)) +
    geom_line() +
    facet_wrap(~usaf_station, scales = 'free_x')
}

# print progress
## note: if the file is already on your system, you'll see no progress bar
(res <- isd(usaf='011690', wban='99999', year=1993, progress=TRUE))

# parallelize processing
# (res <- isd(usaf=172007, wban=99999, year=2016, parallel=TRUE))
}
}
\references{
ftp://ftp.ncdc.noaa.gov/pub/data/noaa/
https://www1.ncdc.noaa.gov/pub/data/noaa
}
\seealso{
Other isd: \code{\link{isd_read}},
  \code{\link{isd_stations_search}},
  \code{\link{isd_stations}}
}
\concept{isd}
